/*
 * Dialog.hpp
 *
 *  Created on: Jan 5, 2010
 *      Author: crueger
 */

#ifndef DIALOG_HPP_
#define DIALOG_HPP_

#include<string>
#include<list>
#include<vector>

#include <boost/filesystem.hpp>

#include "Box.hpp"
#include "LinearAlgebra/Vector.hpp"

class atom;
class Box;
class element;
class molecule;


/** Dialog is one of the two main classes of the UIFactory base class.
 *
 * The Dialog is meant for asking the user for information needed to perform
 * actions he desires, such as asking for a position in space or a length.
 *
 * For this purpose there is the base class Query and numerous specializations
 * for each of the types to be asked. There are primitives integer, doubles and
 * string, but also advanced types such as element, molecule or Vector. There
 * is also an empty query for displaying text.
 */
class Dialog
{
public:
  Dialog();
  virtual ~Dialog();

  template <class T> void query(const char *, std::string = "");

  virtual void queryEmpty(const char *, std::string = "")=0;
  virtual void queryBoolean(const char *, std::string = "")=0;
  virtual void queryInt(const char *, std::string = "")=0;
  virtual void queryInts(const char *, std::string = "")=0;
  virtual void queryDouble(const char*, std::string = "")=0;
  virtual void queryDoubles(const char*, std::string = "")=0;
  virtual void queryString(const char*, std::string = "")=0;
  virtual void queryStrings(const char*, std::string = "")=0;
  virtual void queryAtom(const char*,std::string = "")=0;
  virtual void queryAtoms(const char*,std::string = "")=0;
  virtual void queryMolecule(const char*, std::string = "")=0;
  virtual void queryMolecules(const char*, std::string = "")=0;
  virtual void queryVector(const char*,bool, std::string = "")=0;
  virtual void queryVectors(const char*,bool, std::string = "")=0;
  virtual void queryBox(const char*, std::string = "")=0;
  virtual void queryElement(const char*, std::string = "")=0;
  virtual void queryElements(const char*, std::string = "")=0;
  virtual void queryFile(const char*, std::string = "")=0;

  virtual bool display();

  virtual bool checkAll();
  virtual void setAll();

  virtual bool hasQueries();

protected:
  // methodology for handling queries
  // all queries are stored and then performed at appropriate times

  //these queries can be handled by this dialog

  //TODO: Find a way to reduce complexity...
  //needs O(N*M) query classes, where N is the number of query types and M is the number of GUIs
  //usual approach for reducing inheritance complexity (strategy pattern) does not work,
  //due to lack of common code for query types as well as GUI-Types (all subtypes differ a lot)

  /** Base class for all queries.
   *
   *
   * <h1>How to add another query?</h1>
   *
   * Let's say  we want to query for a type called \a Value.
   *
   * Then, we do the following:
   *  -# Add a class ValueQuery inside class Dialog, the class contains
   *    -# constructor/destructor (latter virtual! because of derived class)
   *    -# virtual bool handle() and virtual void setResult()
   *    -# a protected member tmp of type Value (NOTE: herein the result is stored)
   *    -# if temporaries for conversion are needed put them in here
   *  -# add a function queryValue
   *  -# now, for each of the GUIs we basically have to repeat the above, i.e.
   *     add the class and the function that implement the virtual ones above.
   *    -# QT: an extra class called ValueQtQueryPipe that actually handles
   *       showing dialogs to obtain the value and placing it into the \a tmp
   *       variable (via a given pointer to it as reference). handle() will
   *       simply return true. This is needed because of a restriction of Qt4:
   *       its meta-object-compiler does not like nested classes.
   *    -# CommandLine: nothing special, handle() imports value from \a
   *       CommandLineParser and sets the tmp variable.
   *    -# Text: nothing special, handle() queries the user and sets the tmp
   *       variable
   */
  class Query {
    friend class Dialog;
  public:
    Query(std::string _title, std::string _description = "");
    virtual ~Query();
    virtual bool handle()=0;
    virtual void setResult()=0;
  protected:
    const std::string getTitle() const;
    const std::string getDescription() const;
  private:
    std::string title;  //!< short title of the query
    std::string description; //!< longer description for tooltips or for help
  };

  // Empty Query is just meant for showing text, such as version, help, initial message or alike
  class EmptyQuery : public Query {
  public:
    EmptyQuery(std::string title, std::string _description = "");
    virtual ~EmptyQuery();
    virtual bool handle()=0;
    virtual void setResult();
  };

  //Specialized classes for certain types. GUI-Types are not specialized at this time
  class BooleanQuery : public Query {
  public:
    BooleanQuery(std::string title, std::string _description = "");
    virtual ~BooleanQuery();
    virtual bool handle()=0;
    virtual void setResult();
  protected:
    bool tmp;
  };

  class IntQuery : public Query {
  public:
    IntQuery(std::string title, std::string _description = "");
    virtual ~IntQuery();
    virtual bool handle()=0;
    virtual void setResult();
  protected:
    int tmp;
  };

  class IntsQuery : public Query {
  public:
    IntsQuery(std::string title, std::string _description = "");
    virtual ~IntsQuery();
    virtual bool handle()=0;
    virtual void setResult();
  protected:
    int temp;
    std::vector<int> tmp;
  };

  class DoubleQuery : public Query {
  public:
    DoubleQuery(std::string title, std::string _description = "");
    virtual ~DoubleQuery();
    virtual bool handle()=0;
    virtual void setResult();
  protected:
    double tmp;
  };

  class DoublesQuery : public Query {
  public:
    DoublesQuery(std::string title, std::string _description = "");
    virtual ~DoublesQuery();
    virtual bool handle()=0;
    virtual void setResult();
  protected:
    double temp;
    std::vector<double> tmp;
  };

  class StringQuery : public Query {
  public:
    StringQuery(std::string title, std::string _description = "");
    virtual ~StringQuery();
    virtual bool handle()=0;
    virtual void setResult();
  protected:
    std::string tmp;
  };

  class StringsQuery : public Query {
  public:
    StringsQuery(std::string title, std::string _description = "");
    virtual ~StringsQuery();
    virtual bool handle()=0;
    virtual void setResult();
  protected:
    std::string temp;
    std::vector<std::string> tmp;
  };

  class MoleculeQuery : public Query {
  public:
    MoleculeQuery(std::string title, std::string _description = "");
    virtual ~MoleculeQuery();
    virtual bool handle()=0;
    virtual void setResult();
  protected:
    molecule *tmp;
  };

  class MoleculesQuery : public Query {
  public:
    MoleculesQuery(std::string title, std::string _description = "");
    virtual ~MoleculesQuery();
    virtual bool handle()=0;
    virtual void setResult();
  protected:
    molecule * temp;
    std::vector<molecule *> tmp;
  };

  class AtomQuery : public Query {
  public:
    AtomQuery(std::string title, std::string _description = "");
    virtual ~AtomQuery();
    virtual bool handle()=0;
    virtual void setResult();
  protected:
    atom *tmp;
  };

  class AtomsQuery : public Query {
  public:
    AtomsQuery(std::string title, std::string _description = "");
    virtual ~AtomsQuery();
    virtual bool handle()=0;
    virtual void setResult();
  protected:
    atom *temp;
    std::vector<atom *> tmp;
  };

  class VectorQuery : public Query {
  public:
      VectorQuery(std::string title,bool _check, std::string _description = "");
      virtual ~VectorQuery();
      virtual bool handle()=0;
      virtual void setResult();
    protected:
      Vector tmp;
      bool check;
  };

  class VectorsQuery : public Query {
  public:
      VectorsQuery(std::string title,bool _check, std::string _description = "");
      virtual ~VectorsQuery();
      virtual bool handle()=0;
      virtual void setResult();
    protected:
      Vector temp;
      std::vector<Vector> tmp;
      bool check;
  };

  class BoxQuery : public Query {
  public:
      BoxQuery(std::string title, std::string _description = "");
      virtual ~BoxQuery();
      virtual bool handle()=0;
      virtual void setResult();
    protected:
      Box tmp;
  };

  class ElementQuery : public Query {
  public:
    ElementQuery(std::string title, std::string _description = "");
    virtual ~ElementQuery();
    virtual bool handle()=0;
    virtual void setResult();
  protected:
    const element * tmp;
  };

  class ElementsQuery : public Query {
  public:
    ElementsQuery(std::string title, std::string _description = "");
    virtual ~ElementsQuery();
    virtual bool handle()=0;
    virtual void setResult();
  protected:
    const element *temp;
    std::vector<const element *> tmp;
  };

  class FileQuery : public Query {
  public:
    FileQuery(std::string title, std::string _description = "");
    virtual ~FileQuery();
    virtual bool handle()=0;
    virtual void setResult();
  protected:
    boost::filesystem::path tmp;
  };

void registerQuery(Query* query);

private:
  std::list<Query*> queries;

};


#endif /* DIALOG_HPP_ */
