/*
 * ActionQueue.hpp
 *
 *  Created on: Aug 16, 2013
 *      Author: heber
 */

#ifndef ACTIONQUEUE_HPP_
#define ACTIONQUEUE_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/Singleton.hpp"

#include <vector>

#include "Actions/Action.hpp"
#include "Actions/ActionState.hpp"

namespace MoleCuilder {

class ActionHistory;
class ActionRegistry;
class ActionTrait;

/** This class combines the whole handling of Actions into a single class.
 *
 * It spawns new Actions by its internal ActionRegistry. Spawned Actions are
 * automatically queued and placed into a History after execution.
 */
class ActionQueue : public Singleton<ActionQueue>
{
  friend class Singleton<ActionQueue>;
public:
  typedef std::vector<std::string> ActionTokens_t;
  typedef std::vector< Action * > ActionQueue_t;

  /** Queues the Action with \a name to be called.
   *
   * \param name token of Action to actionqueue
   * \param state whether Actions needs to be filled via a Dialog or not
   */
  void queueAction(const std::string &name, enum Action::QueryOptions state = Action::Interactive);

  /** Queues the Action with \a name to be called.
   *
   * \param _action action to add
   * \param state whether Actions needs to be filled via a Dialog or not
   */
  void queueAction(Action *_action, enum Action::QueryOptions state = Action::Interactive);

  /** Returns the spawned action by token \a name.
   *
   * Action is checked into internal actionqueue.
   *
   * \return pointer to newly spawned action
   */
  Action* getActionByName(const std::string &name);

  /** Checks whether the Action is known by the given \a name.
   *
   * \param name token of action to check
   * \return true - token is known, false - else
   */
  bool isActionKnownByName(const std::string &name) const;

  /** Register an Action with the ActionRegistry.
   *
   * \param _action action to add
   */
  void registerAction(Action *_action);

  /** Returns the vector with the tokens of all currently known Actions.
   *
   * \return list of all tokens
   */
  const ActionTokens_t getListOfActions() const;

  /** Returns the trait to an Action.
   *
   * \param name name of Action
   * \return const ref to its default Trait.
   */
  const ActionTrait& getActionsTrait(const std::string &name) const;

  /** Print the current contents of the actionqueue as CLI instantiated list of Actions.
   *
   * This is useful for storing the current session.
   *
   * \param output output stream to print to
   */
  void outputAsCLI(std::ostream &output) const;

  /** Print the current contents of the actionqueue as Python instantiated list of Actions.
   *
   * This is useful for storing the current session.
   *
   * \param output output stream to print to
   */
  void outputAsPython(std::ostream &output) const;

  /** Undoes last called Acfriend void ::cleanUp();tion.
   *
   */
  void undoLast();

  /** Redoes last undone Action.
   *
   */
  void redoLast();

  /** Getter for the last executed action.
   *
   * \note this is necessary for Reactions to actually have a chance of getting
   * the calculated value as Action's are always cloned.
   *
   * \return const ref to last action
   */
  const Action &getLastAction() const {
    return *(actionqueue.back());
  }

private:
  //!> grant Action access to internal history functions.
  friend class Action;

  /** Wrapper function to add state to ActionHistory.
   *
   * \param _Action Action whose state to add
   * \param _state state to add
   */
  void addElement(Action* _Action, ActionState::ptr _state);

  /** Wrapper function to clear ActionHistory.
   *
   */
  void clear();

  /** Insert an action after CurrentAction.
   *
   * This is implemented only to allow action's COMMAND to work. If we
   * were to use queueAction, actions would come after all other already
   * present actions.
   */
  void insertAction(Action *_action, enum Action::QueryOptions state);

private:
  /** Private cstor for ActionQueue.
   *
   * Must be private as is singleton.
   *
   */
  ActionQueue();

  /** Dstor for ActionQueue.
   *
   */
  ~ActionQueue();

  //!> ActionRegistry to spawn new actions
  ActionRegistry *AR;

  //!> ActionHistory is for undoing and redoing actions, requires ActionRegistry fully initialized
  ActionHistory *history;

  //!> internal actionqueue of actions
  ActionQueue_t actionqueue;

  //!> point to current action in actionqueue
  size_t CurrentAction;
};

};

#endif /* ACTIONQUEUE_HPP_ */
