/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2013 Frederik Heber. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * ActionQueue.cpp
 *
 *  Created on: Aug 16, 2013
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "Actions/ActionQueue.hpp"

#include "CodePatterns/Assert.hpp"
#include "CodePatterns/IteratorAdaptors.hpp"
#include "CodePatterns/Log.hpp"
#include "CodePatterns/Singleton_impl.hpp"

#include <string>
#include <sstream>
#include <vector>

#include "Actions/ActionExceptions.hpp"
#include "Actions/ActionHistory.hpp"
#include "Actions/ActionRegistry.hpp"
#include "World.hpp"

using namespace MoleCuilder;

ActionQueue::ActionQueue() :
    AR(new ActionRegistry()),
    history(new ActionHistory),
    CurrentAction(0)
{}

ActionQueue::~ActionQueue()
{
  // free all actions contained in actionqueue
  for (ActionQueue_t::iterator iter = actionqueue.begin(); !actionqueue.empty(); iter = actionqueue.begin()) {
    delete *iter;
    actionqueue.erase(iter);
  }

  delete history;
  delete AR;
}

void ActionQueue::queueAction(const std::string &name, enum Action::QueryOptions state)
{
  queueAction(AR->getActionByName(name), state);
}

void ActionQueue::queueAction(Action *_action, enum Action::QueryOptions state)
{
  Action *newaction = _action->clone(state);
  newaction->prepare(state);
  actionqueue.push_back( newaction );
  try {
    newaction->call();
  } catch(ActionFailureException &e) {
    std::cerr << "Action " << *boost::get_error_info<ActionNameString>(e) << " has failed." << std::endl;
    World::getInstance().setExitFlag(5);
  }
}

void ActionQueue::insertAction(Action *_action, enum Action::QueryOptions state)
{
  queueAction(_action, state);
}

Action* ActionQueue::getActionByName(const std::string &name)
{
  return AR->getActionByName(name);
}

bool ActionQueue::isActionKnownByName(const std::string &name) const
{
  return AR->isActionPresentByName(name);
}

void ActionQueue::registerAction(Action *_action)
{
  AR->registerInstance(_action);
}

void ActionQueue::outputAsCLI(std::ostream &output) const
{
  for (ActionQueue_t::const_iterator iter = actionqueue.begin();
      iter != actionqueue.end();
      ++iter) {
    // skip store-session in printed list
    if ( ((*iter)->getName() != std::string("store-session"))
        && ((*iter)->getName() != std::string("load-session"))) {
      if (iter != actionqueue.begin())
        output << " ";
      (*iter)->outputAsCLI(output);
    }
  }
  output << std::endl;
}

void ActionQueue::outputAsPython(std::ostream &output) const
{
  const std::string prefix("pyMoleCuilder");
  output << "import " << prefix << std::endl;
  output << "# ========================== Stored Session BEGIN ==========================" << std::endl;
  for (ActionQueue_t::const_iterator iter = actionqueue.begin();
      iter != actionqueue.end();
      ++iter) {
    // skip store-session in printed list
    if ( ((*iter)->getName() != std::string("store-session"))
        && ((*iter)->getName() != std::string("load-session")))
      (*iter)->outputAsPython(output, prefix);
  }
  output << "# =========================== Stored Session END ===========================" << std::endl;
}

const ActionTrait& ActionQueue::getActionsTrait(const std::string &name) const
{
  // this const_cast is just required as long as we have a non-const getActionByName
  const Action * const action = const_cast<ActionQueue *>(this)->getActionByName(name);
  return action->Traits;
}

void ActionQueue::addElement(Action* _Action,ActionState::ptr _state)
{
  history->addElement(_Action, _state);
}

void ActionQueue::clear()
{
  history->clear();
}


const ActionQueue::ActionTokens_t ActionQueue::getListOfActions() const
{
  ActionTokens_t returnlist;

  returnlist.insert(
      returnlist.end(),
      MapKeyConstIterator<ActionRegistry::const_iterator>(AR->getBeginIter()),
      MapKeyConstIterator<ActionRegistry::const_iterator>(AR->getEndIter()));

  return returnlist;
}

void ActionQueue::undoLast()
{
	history->undoLast();
}

void ActionQueue::redoLast()
{
	history->redoLast();
}


CONSTRUCT_SINGLETON(ActionQueue)
