/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * ElementUnitTest.cpp
 *
 *  Created on: Oct 29, 2009
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

using namespace std;

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include <sstream>

#include "Element/element.hpp"

// include headers that implement a archive in simple text format
#include <boost/archive/text_oarchive.hpp>
#include <boost/archive/text_iarchive.hpp>

#include "ElementUnitTest.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( ElementTest );


void ElementTest::setUp()
{
  testelement = new element();
  testelement->setSymbol("H");
};


void ElementTest::tearDown()
{
  delete testelement;
};

/** Test whether assignment works
 *
 */
void ElementTest::AssignmentTest()
{
  element * otherelement = new element();
  element * const backup = otherelement;
  CPPUNIT_ASSERT( otherelement != testelement );
  otherelement = testelement;
  CPPUNIT_ASSERT( otherelement == testelement );
  otherelement = backup;
  *otherelement = *testelement;
  CPPUNIT_ASSERT( *otherelement == *testelement );
  delete backup;

  otherelement = new element(*testelement);
  CPPUNIT_ASSERT( otherelement != testelement );
  CPPUNIT_ASSERT( *otherelement == *testelement );
  delete otherelement;
}

/** Test whether the getters are working.
 *
 */
void ElementTest::GetterTest()
{
  CPPUNIT_ASSERT_EQUAL( (atomId_t)0, testelement->getAtomicNumber() );
  CPPUNIT_ASSERT_EQUAL( 0., testelement->getMass() );
  const unsigned char * color = testelement->getColor();
  for (size_t i = 0; i < 3; ++i)
    CPPUNIT_ASSERT_EQUAL( (unsigned char)0, color[i] );
  CPPUNIT_ASSERT_EQUAL( 0., testelement->getCovalentRadius() );
  CPPUNIT_ASSERT_EQUAL( 0., testelement->getElectronegativity() );
  CPPUNIT_ASSERT_EQUAL( 0., testelement->getVanDerWaalsRadius() );
  CPPUNIT_ASSERT_EQUAL( 0., testelement->getValence() );
  CPPUNIT_ASSERT_EQUAL( 0, testelement->getNoValenceOrbitals() );
  CPPUNIT_ASSERT_EQUAL( 0, testelement->getNoValenceOrbitals() );
  for (size_t i = 0; i < 3; ++i)
    CPPUNIT_ASSERT_EQUAL( 0., testelement->getHBondDistance(i) );
  for (size_t i = 0; i < 3; ++i)
    CPPUNIT_ASSERT_EQUAL( 0., testelement->getHBondAngle(i) );
}

/** Test whether the setters are working.
 *
 */
void ElementTest::SetterTest()
{
  // symbol
  CPPUNIT_ASSERT( testelement->getSymbol() != std::string("He") );
  testelement->setSymbol("He");
  CPPUNIT_ASSERT_EQUAL( std::string("He"), testelement->getSymbol() );

  // name
  CPPUNIT_ASSERT( testelement->getName() != std::string("Helium") );
  testelement->setName("Helium");
  CPPUNIT_ASSERT_EQUAL( std::string("Helium"), testelement->getName() );
}

/** Unit test for operator==()
 */
void ElementTest::ComparisonTest()
{
  testelement->setSymbol("He");
  testelement->setName("Helium");
  element * newelement = new element();
  newelement->setSymbol("H");
  newelement->setName("Hydrogen");

  CPPUNIT_ASSERT( *testelement != *newelement );

  newelement->setSymbol("He");
  newelement->setName("Helium");

  CPPUNIT_ASSERT( *testelement == *newelement );

  delete newelement;
}

/** Tests whether serialization of the class works
 */
void ElementTest::SerializeTest()
{
  const std::string he("He");
  const std::string helium("Helium");
  testelement->setSymbol(he);
  testelement->setName(helium);

  CPPUNIT_ASSERT_EQUAL ( he, testelement->getSymbol() );
  CPPUNIT_ASSERT_EQUAL( helium, testelement->getName() );

  // write element to stream
  std::stringstream stream;
  boost::archive::text_oarchive oa(stream);
  oa << testelement;

  std::cout << "Contents of archive is " << stream.str() << std::endl;

  // create and open an archive for input
  boost::archive::text_iarchive ia(stream);
  // read class state from archive
  element * newelement;

  ia >> newelement;

  CPPUNIT_ASSERT_EQUAL( he, newelement->getSymbol() );
  CPPUNIT_ASSERT_EQUAL( helium, newelement->getName() );

  CPPUNIT_ASSERT (*testelement == *newelement);

  delete newelement;
};
