/*
 * FunctionModel.hpp
 *
 *  Created on: 02.10.2012
 *      Author: heber
 */

#ifndef FUNCTIONMODEL_HPP_
#define FUNCTIONMODEL_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <boost/function.hpp>
#include <list>
#include <vector>

#include "FunctionApproximation/FunctionArgument.hpp"

class Fragment;
class HomologyGraph;
class TrainingData;

/** This class represents the interface for a given function to model a
 * high-dimensional data set in FunctionApproximation.
 *
 * As the parameters may be stored differently, the interface functions for
 * getting and setting them are as light-weight (and not speed-optimized)
 * as possible.
 *
 * We always work in distances, i.e. pairs of atoms and the distance in between.
 * As fragments do not contain these distances directly but the atomic positions
 * (and charges) instead, we need to extract these from the fragment. For this
 * purpose we need a bound function, termed an 'Extractor'. However, this is only
 * required when one wants to use a FunctionModel directly on a given fragment.
 * In FunctionApproximation we instead have TrainingData generate automatically
 * a list of all pair-wise distances. The FunctionModel's Extractor may however
 * create a more specific (and tighter) list of arguments, which however can
 * then only be used with this specific FunctionModel.
 *
 * Furthermore, the underlying function to fit may require these distances, or
 * arguments (termed so if paired with charges and atomic indices), to be in a
 * certain order or does need only a subset. For this purpose we need another
 * bound function, called a 'Filter'.
 *
 * As a fragment may contain multiple sets of arguments or distances that serve
 * as valid function arguments, we need to split these sets up, such that they
 * can be served one by one to the function. For this purpose we need a function
 * that gives the number of arguments per set. (note that the Filter is supposed
 * to place related arguments consecutively.
 *
 */
class FunctionModel
{
public:
  //!> typedef for a single parameter degree of freedom of the function
  typedef double parameter_t;
  //!> typedef for the whole set of parameters of the function
  typedef std::vector<parameter_t> parameters_t;
  //!> typedef for the argument vector as input to the function (subset of distances)
  typedef std::vector<argument_t> arguments_t;
  //!> typedef for a list of argument vectors as input to the function (list of subsets)
  typedef std::list<arguments_t> list_of_arguments_t;
  //!> typedef for a single result degree of freedom
  typedef double result_t;
  //!> typedef for the result vector as returned by the function
  typedef std::vector<result_t> results_t;
  //!> typedef for a function containing how to extract required information from a Fragment.
  typedef boost::function< list_of_arguments_t (const Fragment &, const size_t)> extractor_t;
  //!> typedef for a function containing how to filter required distances from a full argument list.
  typedef boost::function< list_of_arguments_t (const HomologyGraph &, const arguments_t &)> filter_t;
  //!> typedef for the magic triple function that gets the other two distances for a given argument
  typedef boost::function< std::vector<arguments_t>(const argument_t &, const double)> triplefunction_t;

public:
  FunctionModel() {}
  virtual ~FunctionModel() {}

  /** Setter for the parameters of the model function.
   *
   * \param params set of parameters to set
   */
  virtual void setParameters(const parameters_t &params)=0;

  /** Getter for the parameters of this model function.
   *
   * \return current set of parameters of the model function
   */
  virtual parameters_t getParameters() const=0;

  /** Sets the parameter randomly within the sensible range of each parameter.
   *
   * \param data container with training data for guesstimating range
   */
  virtual void setParametersToRandomInitialValues(const TrainingData &data)=0;

  /** Getter for the number of parameters of this model function.
   *
   * \return number of parameters
   */
  virtual size_t getParameterDimension() const=0;

  /** Sets the magic triple function that we use for getting angle distances.
   *
   * @param _triplefunction function that returns a list of triples (i.e. the
   *        two remaining distances) to a given pair of points (contained as
   *        indices within the argument)
   */
  virtual void setTriplefunction(triplefunction_t &_triplefunction)
  {}

  /** Evaluates the function with the given \a arguments and the current set of
   * parameters.
   *
   * \param arguments set of arguments as input variables to the function
   * \return result of the function
   */
  virtual results_t operator()(const list_of_arguments_t &arguments) const=0;

  /** Evaluates the derivative of the function with the given \a arguments
   * with respect to a specific parameter indicated by \a index.
   *
   * \param arguments set of arguments as input variables to the function
   * \param index derivative of which parameter
   * \return result vector containing the derivative with respect to the given
   *         input
   */
  virtual results_t parameter_derivative(const list_of_arguments_t &arguments, const size_t index) const=0;

  /** States whether lower and upper boundaries should be used to constraint
   * the parameter search for this function model.
   *
   * \return true - constraints should be used, false - else
   */
  virtual bool isBoxConstraint() const=0;

  /** Returns a vector which are the lower boundaries for each parameter_t
   * of this FunctionModel.
   *
   * \return vector of parameter_t resembling lowest allowed values
   */
  virtual parameters_t getLowerBoxConstraints() const=0;

  /** Returns a vector which are the upper boundaries for each parameter_t
   * of this FunctionModel.
   *
   * \return vector of parameter_t resembling highest allowed values
   */
  virtual parameters_t getUpperBoxConstraints() const=0;

  /** Returns a bound function to be used with TrainingData, extracting distances
   * from a Fragment.
   *
   * \return bound function extracting distances from a fragment
   */
  virtual filter_t getSpecificFilter() const=0;

  /** Returns the number of arguments the underlying function requires.
   *
   * \return number of arguments of the function
   */
  virtual size_t getSpecificArgumentCount() const=0;
};

#endif /* FUNCTIONMODEL_HPP_ */
