/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * AnalysisCorrelationToPointUnitTest.cpp
 *
 *  Created on: Oct 13, 2009
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

using namespace std;

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include <cstring>

#include "analysis_correlation.hpp"

#include "Descriptors/MoleculeDescriptor.hpp"

#include "atom.hpp"
#include "element.hpp"
#include "molecule.hpp"
#include "linkedcell.hpp"
#include "periodentafel.hpp"
#include "World.hpp"

#include "AnalysisCorrelationToPointUnitTest.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( AnalysisCorrelationToPointUnitTest );

void AnalysisCorrelationToPointUnitTest::setUp()
{
  atom *Walker = NULL;

  // init private all pointers to zero
  TestMolecule = NULL;
  pointmap = NULL;
  binmap = NULL;
  point = NULL;

  // construct element list
  std::vector<const element *> elements;
  hydrogen = World::getInstance().getPeriode()->FindElement(1);
  CPPUNIT_ASSERT(hydrogen != NULL && "hydrogen element not found");
  elements.push_back(hydrogen);
  // construct molecule (tetraeder of hydrogens)
  TestMolecule = World::getInstance().createMolecule();
  Walker = World::getInstance().createAtom();
  Walker->setType(hydrogen);
  Walker->setPosition(Vector(1., 0., 1. ));
  TestMolecule->AddAtom(Walker);
  Walker = World::getInstance().createAtom();
  Walker->setType(hydrogen);
  Walker->setPosition(Vector(0., 1., 1. ));
  TestMolecule->AddAtom(Walker);
  Walker = World::getInstance().createAtom();
  Walker->setType(hydrogen);
  Walker->setPosition(Vector(1., 1., 0. ));
  TestMolecule->AddAtom(Walker);
  Walker = World::getInstance().createAtom();
  Walker->setType(hydrogen);
  Walker->setPosition(Vector(0., 0., 0. ));
  TestMolecule->AddAtom(Walker);

  // check that TestMolecule was correctly constructed
  CPPUNIT_ASSERT_EQUAL( TestMolecule->getAtomCount(), 4 );

  TestMolecule->ActiveFlag = true;

  // init point
  point = new Vector(1.,1.,1.);

  // init maps
  World::getInstance().selectAllMolecules(AllMolecules());
  allMolecules = World::getInstance().getSelectedMolecules();
  CPPUNIT_ASSERT_EQUAL( (size_t) 1, allMolecules.size());
  pointmap = CorrelationToPoint( allMolecules, elements, (const Vector *)point );
  binmap = NULL;

};


void AnalysisCorrelationToPointUnitTest::tearDown()
{
  if (pointmap != NULL)
    delete(pointmap);
  if (binmap != NULL)
    delete(binmap);

  delete(point);
  World::purgeInstance();
  logger::purgeInstance();
};

void AnalysisCorrelationToPointUnitTest::CorrelationToPointTest()
{
  // do the pair correlation
  CPPUNIT_ASSERT( pointmap != NULL );
  CPPUNIT_ASSERT_EQUAL( (size_t)4, pointmap->size() );

};

void AnalysisCorrelationToPointUnitTest::CorrelationToPointBinNoRangeTest()
{
  BinPairMap::iterator tester;
  // put pair correlation into bins and check with no range
  binmap = BinData( pointmap, 0.5, 0., 0. );
  OutputCorrelationMap<BinPairMap> ( (ofstream *)&cout, binmap, OutputCorrelation_Header, OutputCorrelation_Value );
  CPPUNIT_ASSERT_EQUAL( (size_t)3, binmap->size() );
  tester = binmap->begin();
  CPPUNIT_ASSERT_EQUAL( 1., tester->first );
  CPPUNIT_ASSERT_EQUAL( 3, tester->second );

};

void AnalysisCorrelationToPointUnitTest::CorrelationToPointBinRangeTest()
{
  BinPairMap::iterator tester;
  // ... and check with [0., 2.] range
  binmap = BinData( pointmap, 0.5, 0., 2. );
  OutputCorrelationMap<BinPairMap> ( (ofstream *)&cout, binmap, OutputCorrelation_Header, OutputCorrelation_Value );
  CPPUNIT_ASSERT_EQUAL( (size_t)5, binmap->size() );
  tester = binmap->begin();
  CPPUNIT_ASSERT_EQUAL( 0., tester->first );
  tester = binmap->find(1.);
  CPPUNIT_ASSERT_EQUAL( 1., tester->first );
  CPPUNIT_ASSERT_EQUAL( 3, tester->second );

};
