/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * PairPotential_MorseUnitTest.cpp
 *
 *  Created on: Oct 04, 2012
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

using namespace std;

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include "PairPotential_MorseUnitTest.hpp"

#include <boost/assign.hpp>

#include "CodePatterns/Assert.hpp"

#include "FunctionApproximation/FunctionArgument.hpp"
#include "Potentials/Specifics/PairPotential_Morse.hpp"
#include "Potentials/helpers.hpp"

using namespace boost::assign;

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( PairPotential_MorseTest );


void PairPotential_MorseTest::setUp()
{
  // failing asserts should be thrown
  ASSERT_DO(Assert::Throw);

  // data is taken from gnuplot via
  // g(x)=D*(1- exp(-a*(x-c)))**2+d
  // set table "morse.dat"
  // plot [1.89012:4.46595] g(x)
  a = 0.897888;
  c = 2.92953;
  d = -78.9883;
  D = 0.196289;
  input +=
      1.89012,
      2.17632,
      2.46253,
      2.74873,
      3.03493,
      3.32114,
      3.60734,
      3.89354,
      4.17974,
      4.46595;
  output +=
			0.467226,
			0.183388,
			0.0532649,
			0.00609808,
			0.00160056,
			0.0172506,
			0.0407952,
			0.0658475,
			0.0893157,
			0.109914;

  CPPUNIT_ASSERT_EQUAL( input.size(), output.size() );
}


void PairPotential_MorseTest::tearDown()
{
}

/** UnitTest for operator()
 */
void PairPotential_MorseTest::operatorTest()
{
  PairPotential_Morse::ParticleTypes_t types =
      boost::assign::list_of<PairPotential_Morse::ParticleType_t>
        (0)(1)
      ;
  PairPotential_Morse Morse(types, a,c,D);
  for (size_t index = 0; index < input.size(); ++index) {
    argument_t arg(argument_t::indices_t(0,1), argument_t::types_t(0,1), input[index]);
    CPPUNIT_ASSERT(
        Helpers::isEqual(
            output[index],
            Morse( FunctionModel::arguments_t(1,arg) )[0],
            1.e-4/std::numeric_limits<double>::epsilon() // only compare four digits
        )
    );
  }
}

/** UnitTest for derivative()
 */
void PairPotential_MorseTest::derivativeTest()
{
  PairPotential_Morse::ParticleTypes_t types =
      boost::assign::list_of<PairPotential_Morse::ParticleType_t>
        (0)(1)
      ;
  PairPotential_Morse Morse(types, a,c,D);
  argument_t arg(argument_t::indices_t(0,1), argument_t::types_t(0,1), c);
  CPPUNIT_ASSERT(
      Helpers::isEqual(
          0.,
          Morse.derivative(
              FunctionModel::arguments_t(1,arg)
          )[0],
          1.e+6
      )
  );
}


/** UnitTest for parameter_derivative()
 */
void PairPotential_MorseTest::parameter_derivativeTest()
{
  PairPotential_Morse::ParticleTypes_t types =
      boost::assign::list_of<PairPotential_Morse::ParticleType_t>
        (0)(1)
      ;
  PairPotential_Morse Morse(types, a,c,D);
  argument_t arg(argument_t::indices_t(0,1), argument_t::types_t(0,1), c);
  CPPUNIT_ASSERT(
      Helpers::isEqual(
          0.,
          Morse.parameter_derivative(
              FunctionModel::arguments_t(1,arg),
              0
          )[0],
          1.e+6
      )
  );
  CPPUNIT_ASSERT(
      Helpers::isEqual(
          0.,
          Morse.parameter_derivative(
              FunctionModel::arguments_t(1,arg),
              1
          )[0],
          1.e+6
      )
  );
  CPPUNIT_ASSERT(
      Helpers::isEqual(
          0.,
          Morse.parameter_derivative(
              FunctionModel::arguments_t(1,arg),
              2
          )[0],
          1.e+6
      )
  );
}
