/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * Observable.cpp
 *
 *  Created on: Dec 1, 2011
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "MemDebug.hpp"

#include "Observable.hpp"

#include "Assert.hpp"
#include "Channels.hpp"
#include "defs.hpp"
#include "Notification.hpp"


// All infrastructure for the observer-pattern is bundled at a central place
// this is more efficient if many objects can be observed (inherit from observable)
// but only few are actually coupled with observers. E.g. TMV has over 500.000 Atoms,
// which might become observable. Handling Observable infrastructure in each of
// these would use memory for each atom. By handling Observer-infrastructure
// here we only need memory for objects that actually are observed.
// See [Gamma et al, 1995] p. 297

std::map<Observable*, int> Observable::depth;  //!< Map of Observables to the depth of the DAG of Observers
std::map<Observable*,std::multimap<int,Observer*> > Observable::callTable; //!< Table for each Observable of all its Observers
std::map<Observable*,std::set<Notification*> > Observable::notifications; //!< Table for all current notifications to perform
std::set<Observable*> Observable::busyObservables; //!< Set of Observables that are currently busy notifying their sign-on'ed Observers
Observable::ChannelMap Observable::NotificationChannels; //!< Map of Observables to their Channels.

// ValidRange must be initialized before PriorityLevel.
range<int> Observable::PriorityLevel::ValidRange(-20, 21);
Observable::PriorityLevel Observable::PriorityDefault(int(0));

/** Constructor of PriorityLevel.
 *
 * \note We check whether the level is within Observable::PriorityLevel::ValidRange.
 *
 * @param i priority level encapsulated in this class.
 */
Observable::PriorityLevel::PriorityLevel(const int i) :
    level(i)
{
  ASSERT(ValidRange.isInRange(level),
      "Observable::PriorityLevel::PriorityLevel() - Priority level "
      +toString(level)+" out of range "+toString(ValidRange)+".");
}

Observable::PriorityLevel::~PriorityLevel()
{}

/** Attaching Sub-observables to Observables.
 * Increases entry in Observable::depth for this \a *publisher by one.
 *
 * The two functions \sa start_observer_internal() and \sa finish_observer_internal()
 * have to be used together at all time. Never use these functions directly
 * START_OBSERVER and FINISH_OBSERVER also construct a bogus while(0) loop
 * thus producing compiler-errors whenever only one is used.
 * \param *publisher reference of sub-observable
 */
void Observable::start_observer_internal(Observable *publisher){
  // increase the count for this observable by one
  // if no entry for this observable is found, an new one is created
  // by the STL and initialized to 0 (see STL documentation)
#ifdef LOG_OBSERVER
  observerLog().addMessage(depth[publisher]) << ">> Locking " << observerLog().getName(publisher) << std::endl;
#endif
  depth[publisher]++;
}

/** Detaching Sub-observables from Observables.
 * Decreases entry in Observable::depth for this \a *publisher by one. If zero, we
 * start notifying all our Observers.
 *
 * The two functions start_observer_internal() and finish_observer_internal()
 * have to be used together at all time. Never use these functions directly
 * START_OBSERVER and FINISH_OBSERVER also construct a bogus while(0) loop
 * thus producing compiler-errors whenever only one is used.
 * \param *publisher reference of sub-observable
 */
void Observable::finish_observer_internal(Observable *publisher){
  // decrease the count for this observable
  // if zero is reached all observed blocks are done and we can
  // start to notify our observers
  --depth[publisher];
#ifdef LOG_OBSERVER
  observerLog().addMessage(depth[publisher]) << "<< Unlocking " << observerLog().getName(publisher) << std::endl;
#endif
  if(depth[publisher]){}
  else{
    publisher->notifyAll();
    // this item is done, so we don't have to keep the count with us
    // save some memory by erasing it
    depth.erase(publisher);
  }
}

void Observable::enque_notification_internal(Observable *publisher, Notification_ptr notification){
  notifications[publisher].insert(notification);
}

/** Constructor for Observable Protector.
 * Basically, calls start_observer_internal(). Hence use this class instead of
 * calling the function directly.
 *
 * \param *protege Observable to be protected.
 */
Observable::_Observable_protector::_Observable_protector(Observable *_protege) :
  protege(_protege)
{
  start_observer_internal(protege);
}

Observable::_Observable_protector::_Observable_protector(const _Observable_protector &dest) :
    protege(dest.protege)
{
  start_observer_internal(protege);
}

/** Destructor for Observable Protector.
 * Basically, calls finish_observer_internal(). Hence use this class instead of
 * calling the function directly.
 *
 * \param *protege Observable to be protected.
 */
Observable::_Observable_protector::~_Observable_protector()
{
  finish_observer_internal(protege);
}

/************* Notification mechanism for observables **************/

/** Notify all Observers of changes.
 * Puts \a *this into Observable::busyObservables, calls Observer::update() for all in callee_t
 * and removes from busy list.
 */
void Observable::notifyAll() {
  // we are busy notifying others right now
  // add ourselves to the list of busy subjects to enable circle detection
  busyObservables.insert(this);
  // see if anyone has signed up for observation
  // and call all observers
  try {
    if(callTable.count(this)) {
      // elements are stored sorted by keys in the multimap
      // so iterating over it gives us a the callees sorted by
      // the priorities
      callees_t callees = callTable[this];
      callees_t::iterator iter;
      for(iter=callees.begin();iter!=callees.end();++iter){
#ifdef LOG_OBSERVER
        observerLog().addMessage() << "-> Sending update from " << observerLog().getName(this)
                                   << " to " << observerLog().getName((*iter).second)
                                   << " (priority=" << (*iter).first << ")"<< std::endl;
#endif
        (*iter).second->update(this);
      }
    }
  }
  ASSERT_NOCATCH("Exception thrown from Observer Update");

  // send out all notifications that need to be done

  notificationSet currentNotifications = notifications[this];
  for(notificationSet::iterator it = currentNotifications.begin();
      it != currentNotifications.end();++it){
    (*it)->notifyAll(this);
  }

  notifications.erase(this);

   // done with notification, we can leave the set of busy subjects
  busyObservables.erase(this);
}


/** Handles passing on updates from sub-Observables.
 * Mimicks basically the Observer::update() function.
 *
 * \param *publisher The \a *this we observe.
 */
void Observable::update(Observable *publisher) {
  // circle detection
  if(busyObservables.find(this)!=busyObservables.end()) {
    // somehow a circle was introduced... we were busy notifying our
    // observers, but still we are called by one of our sub-Observables
    // we cannot be sure observation will still work at this point
    ASSERT(0,"Circle detected in observation-graph.\n"
             "Observation-graph always needs to be a DAG to work correctly!\n"
             "Please check your observation code and fix this!\n");
    return;
  }
  else {
    // see if we are in the process of changing ourselves
    // if we are changing ourselves at the same time our sub-observables change
    // we do not need to publish all the changes at each time we are called
    if(depth.find(this)==depth.end()) {
#ifdef LOG_OBSERVER
      observerLog().addMessage() << "-* Update from " << observerLog().getName(publisher)
                                 << " propagated by " << observerLog().getName(this) << std::endl;
#endif
      notifyAll();
    }
    else{
#ifdef LOG_OBSERVER
      observerLog().addMessage() << "-| Update from " <<  observerLog().getName(publisher)
                                 << " not propagated by " << observerLog().getName(this) << std::endl;
#endif
    }
  }
}

/** Sign on an Observer to this Observable.
 * Puts \a *target into Observable::callTable list.
 * \param *target Observer
 * \param priority number in [-20,20]
 */
void Observable::signOn(Observer *target, PriorityLevel priority) const
{
#ifdef LOG_OBSERVER
  observerLog().addMessage() << "@@ Signing on " << observerLog().getName(target) << " to " << observerLog().getName(const_cast<Observable *>(this)) << std::endl;
#endif
  bool res = false;
  callees_t &callees = callTable[const_cast<Observable *>(this)];

  callees_t::iterator iter;
  for(iter=callees.begin();iter!=callees.end();++iter){
    res |= ((*iter).second == target);
  }
  if(!res)
    callees.insert(std::pair<int,Observer*>(priority.level,target));
}

/** Sign off an Observer from this Observable.
 * Removes \a *target from Observable::callTable list.
 * \param *target Observer
 */
void Observable::signOff(Observer *target) const
{
  ASSERT(callTable.count(const_cast<Observable *>(this)),
      "SignOff called for an Observable without Observers.");
#ifdef LOG_OBSERVER
  observerLog().addMessage() << "** Signing off " << observerLog().getName(target) << " from " << observerLog().getName(const_cast<Observable *>(this)) << std::endl;
#endif
  callees_t &callees = callTable[const_cast<Observable *>(this)];

  callees_t::iterator iter;
  callees_t::iterator deliter;
  for(iter=callees.begin();iter!=callees.end();) {
    if((*iter).second == target) {
      callees.erase(iter++);
    }
    else {
      ++iter;
    }
  }
  if(callees.empty()){
    callTable.erase(const_cast<Observable *>(this));
  }
}

void Observable::signOn(Observer *target, Notification_ptr notification) const
{
  notification->addObserver(target);
}

void Observable::signOff(Observer *target, Notification_ptr notification) const
{
  notification->removeObserver(target);
}

bool Observable::isBlocked() const
{
  return depth.count(const_cast<Observable *>(this)) > 0;
}

Notification_ptr Observable::getChannel(size_t no) const
{
  const ChannelMap::const_iterator iter = NotificationChannels.find(const_cast<Observable * const>(this));
  ASSERT(iter != NotificationChannels.end(),
      "Observable::getChannel() - we do not have a channel in NotificationChannels.");
  const Channels *OurChannel = iter->second;
  ASSERT(OurChannel != NULL,
      "Observable::getChannel() - observable has no channels.");
  return OurChannel->getChannel(no);
}

/** Handles sub-observables that just got killed
 *  when an sub-observerable dies we usually don't need to do anything
 *  \param *publisher Sub-Observable.
 */
void Observable::subjectKilled(Observable *publisher){
}

/** Constructor for class Observable.
 */
Observable::Observable(std::string name) :
  Observer(Observer::BaseConstructor())
{
#ifdef LOG_OBSERVER
  observerLog().addName(this,name);
  observerLog().addMessage() << "++ Creating Observable " << observerLog().getName(this) << std::endl;
#endif
}

/** Destructor for class Observable.
 * When an observable is deleted, we let all our observers know. \sa Observable::subjectKilled().
 */
Observable::~Observable()
{
#ifdef LOG_OBSERVER
  observerLog().addMessage() << "-- Destroying Observable " << observerLog().getName(this) << std::endl;
#endif
  if(callTable.count(this)) {
    // delete all entries for this observable
    callees_t callees = callTable[this];
    callees_t::iterator iter;
    for(iter=callees.begin();iter!=callees.end();++iter){
      (*iter).second->subjectKilled(this);
    }
    callTable.erase(this);
  }
}
