/*
 * Menu.h
 *
 *  Created on: Dec 10, 2009
 *      Author: crueger
 */

#ifndef MENU_MENU_H_
#define MENU_MENU_H_

#include <set>
#include <string>

#include "Menu/MenuInterface.hpp"
#include "Menu/MenuDescription.hpp"

/** Base class for all Types of menus.
 * Here, we simply initialize the menus. Via the MenuInterface wrapper we may
 * access the adding of items for each specific menu in a uniform manner.
 *
 * Note that this Class is never to be used directly but only via derived
 * specializations.
 *
 * Note that we do not call member function init() directly in the constructor
 * to allow for something to be add to the menu beforehand which is necessary
 * for the TextMenu for example.
 *
 * <h1>Howto</h1>
 *
 * First, derive your own menu class, see QtMenu or TextMenu for example. These
 * have been templated because of the two Qt classes QMenu and QMenuBar to be
 * addressable in a unified manner. TextMenu does not actually need to be a
 * template but it does not hurt either.
 *
 * Basically, the structure is as follows:
 *  -# Menu implements the initialization: it goes through MenuDescription's and
 *     ActionRegistry's contents and adds MenuItem's and Action's as requested.
 *  -# It does so by calling virtual functions defined in MenuInterface. These
 *     allow for adding of specific type of items: separators, Actions, menus
 *  -# Your derived Menu implements add these specific adders, basically just as
 *     wrappers. If you really have another UI type, implement its Menu
 *     functionality separately, such as TxMenu does.
 *  -# It also inherits both Menu and virtually MenuInterface such that it
 *     contains initializing and adding functionality
 *
 *  Note that MenuInterface is inherited by both Menu and your class and hence has
 *  to be declared as a "virtual" base class for both.
 */
class Menu : virtual public MenuInterface
{
public:
  explicit Menu(const std::string &name);
  virtual ~Menu();

  void init();

protected:
  // Unique name of the menu for identification.
  const std::string name;

  // populater function that adds all menu items
  void populate();
  void populateActions();

private:
  void addAction(const std::string &ActionName);
  void addSeparator();
  void addSubmenu(const std::string &MenuName, const int MenuPosition);
  bool isPresent(const std::string &token);

  enum ItemType {ActionItem, MenuItem, SeparatorItem, NoItem};

  int TopPosition;  // current position to add
  MenuDescription menudescriptions; // contains the menu tree and description to each item
  enum ItemType LastItem;  // check whether separator followed separator
  std::set <std::string> DuplicatesList;  // is used to check for duplicates
};

#endif /* MENU_H_ */
