/*
 * Project: JobMarket
 * Description: asynchronous Server/Controller/Client-approach to parallel computing, based on boost::asio
 * Copyright (C)  2010 Frederik Heber. All rights reserved.
 * 
 */

/*
 * SyncOperation.cpp
 *
 *  Created on: Mar 04, 2012
 *      Author: heber
 */


// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// boost asio needs specific operator new
#include <boost/asio.hpp>

#include "CodePatterns/MemDebug.hpp"

#include "JobMarket/Operations/SyncOperation.hpp"

#include <boost/bind.hpp>
#include <vector>
#include "JobMarket/Connection.hpp" // Must come before boost/serialization headers.
#include "CodePatterns/Info.hpp"
#include "CodePatterns/Log.hpp"

/** Internal function to connect connection_.
 *
 */
void SyncOperation::connect(const std::string& _host, const std::string& _service)
{
  // Resolve the host name into an IP address.
  boost::asio::ip::tcp::resolver resolver(connection_.socket().get_io_service());
  boost::asio::ip::tcp::resolver::query query(_host, _service);
  boost::asio::ip::tcp::resolver::iterator endpoint_iterator =
    resolver.resolve(query);
  // check whether host could be resolved
  if (endpoint_iterator != boost::asio::ip::tcp::resolver::iterator()) {
    boost::asio::ip::tcp::endpoint endpoint = *endpoint_iterator;

    // Start an asynchronous connect operation.
    LOG(3, "DEBUG: Connecting synchronously to endpoint " << endpoint << " ...");
    connection_.socket().connect(endpoint);
  } else {
    status = Operation::error;
  }
}


/** Internal function to disconnect connection_ correctly.
 *
 */
void SyncOperation::disconnect()
{
  connection_.socket().close();
}

/** Wrapper function for the virtual call to internal() that connects and disconnects.
 *
 * @param _host host address to connect to
 * @param _service service to connect to
 */
void SyncOperation::operator()(const std::string& _host, const std::string& _service)
{
  DEBUG_FUNCTION_ENTRYEXIT

  status = Operation::running;

  // connect
  connect(_host, _service);

  // call virtual function to continue
  internal();

  // disconnect
  disconnect();

  if (status == Operation::running)
    status = Operation::success;
}
