/*
 * Project: JobMarket
 * Description: asynchronous Server/Controller/Client-approach to parallel computing, based on boost::asio
 * Copyright (C)  2011 Frederik Heber. All rights reserved.
 * 
 */

/*
 * AsyncOperation.cpp
 *
 *  Created on: Nov 11, 2011
 *      Author: heber
 */


// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// boost asio needs specific operator new
#include <boost/asio.hpp>

#include "CodePatterns/MemDebug.hpp"

#include "JobMarket/Operations/AsyncOperation.hpp"

#include <boost/bind.hpp>
#include <boost/function.hpp>
#include <vector>
#include "JobMarket/Connection.hpp" // Must come before boost/serialization headers.
#include "CodePatterns/Info.hpp"
#include "CodePatterns/Log.hpp"

static void NoOp() {}

// static instances
const boost::function<void ()> AsyncOperation::NoOpCallback = boost::bind(&NoOp);

/** Callback function when an operation has been completed.
 *
 * \param e error code if something went wrong
 */
void AsyncOperation::handle_FinishOperation(const boost::system::error_code& e)
{
  DEBUG_FUNCTION_ENTRYEXIT
  if (!e)
  {
    LOG(1, "INFO: AsyncOperation completed.");
    status = Operation::success;
    callback_on_success();
  }
  else
  {
    // An error occurred.
    ELOG(1, e.message());
    status = Operation::error;
    callback_on_failure();
  }

  // Since we are not starting a new operation the io_service will run out of
  // work to do and the client will exit.
  disconnect();
}

/** Internal function to disconnect connection_ correctly.
 *
 */
void AsyncOperation::disconnect()
{
  OBSERVE;
  // close the socket
  connection_.socket().close();
}

void AsyncOperation::operator()(const std::string& _host, const std::string& _service)
{
  DEBUG_FUNCTION_ENTRYEXIT

  status = Operation::running;

  // Resolve the host name into an IP address.
  boost::asio::ip::tcp::resolver resolver(connection_.socket().get_io_service());
  boost::asio::ip::tcp::resolver::query query(_host, _service);
  boost::system::error_code ec;
  boost::asio::ip::tcp::resolver::iterator endpoint_iterator =
    resolver.resolve(query, ec);
  // check whether host could be resolved
  if (endpoint_iterator != boost::asio::ip::tcp::resolver::iterator()) {
    boost::asio::ip::tcp::endpoint endpoint = *endpoint_iterator;

    // Start an asynchronous connect operation.
    LOG(3, "DEBUG: Connecting asynchronously to endpoint " << endpoint << " ...");
    connection_.socket().async_connect(endpoint,
      boost::bind(&AsyncOperation::handle_connect, this,
        boost::asio::placeholders::error, ++endpoint_iterator));
  } else {
    // this will allow to still listen on the socket for other connections
    handle_FinishOperation(ec);
  }
}
