/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C) 2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * IndexSetContainerUnitTest.cpp
 *
 *  Created on: Jul 3, 2012
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

using namespace std;

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include "Fragmentation/Summation/IndexSetContainer.hpp"

#include "IndexSetContainerUnitTest.hpp"

#include <boost/assign.hpp>
#include <boost/foreach.hpp>

#include "Fragmentation/KeySetsContainer.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

using namespace boost::assign;

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( IndexSetContainerTest );


void IndexSetContainerTest::setUp()
{
  allsets.reserve(5);
  IndexSet tempset;
  tempset += 1;
  allsets += tempset;
  tempset += 2;
  allsets += tempset;
  tempset.clear();
  tempset += 3;
  allsets += tempset;
  tempset += 2;
  allsets += tempset;
  tempset += 4;
  allsets += tempset;
  tempset += 1;
  allsets += tempset;
  CPPUNIT_ASSERT_EQUAL( (size_t)6, allsets.size() );

  SV = NULL;
}


void IndexSetContainerTest::tearDown()
{
  allsets.clear();
  delete SV;
}

/** UnitTest for constructor
 */
void IndexSetContainerTest::constructorTest()
{
  SV = new IndexSetContainer(allsets);
  // check that container is sorted
  const IndexSetContainer::Container_t &container = SV->getContainer();
//  BOOST_FOREACH( IndexSet_ptr a, container) {
//    std::cout << *a << "\t";
//  }
//  std::cout << endl;
  CPPUNIT_ASSERT_EQUAL( (size_t)6, container.size() );
}


/** UnitTest for operator==,!=,<,>()
 */
void IndexSetContainerTest::comparatorTest()
{
  SV = new IndexSetContainer(allsets);
  // check that container is sorted
  typedef IndexSetContainer::Container_t::const_iterator iterator;
  const IndexSetContainer::Container_t &container = SV->getContainer();
  iterator frontiter = ++(container.begin());
  for (iterator backiter = container.begin();
      frontiter != container.end(); ++backiter, ++frontiter) {
    //std::cout << **backiter << " < " << **frontiter << std::endl;
    CPPUNIT_ASSERT( **backiter < **frontiter );
  }
}

/** Unit tests for insert()
 *
 */
void IndexSetContainerTest::insertTest()
{
  SV = new IndexSetContainer(allsets);
  IndexSet_ptr tempset(new IndexSet);
  (*tempset) += 1,2,3,4;

  // insert present set and check that it is still sorted
  CPPUNIT_ASSERT( SV->ContainerSorted );
  SV->insert(tempset);
  CPPUNIT_ASSERT( SV->ContainerSorted );

  // insert new set and check that it is now unsorted
  (*tempset) += 5;
  CPPUNIT_ASSERT( SV->ContainerSorted );
  SV->insert(tempset);
  CPPUNIT_ASSERT( !SV->ContainerSorted );
}

/** Unit tests for cstr(KeySetsContainer)
 *
 */
void IndexSetContainerTest::KeySetContainerTest()
{
  KeySetsContainer keysets;
  KeySetsContainer::IntVector vector1;
  vector1 += 1,-1;
  KeySetsContainer::IntVector vector2;
  vector2 += -1,1,2,-1;
  keysets.KeySets.push_back(vector1);
  keysets.KeySets.push_back(vector2);
  SV = new IndexSetContainer(keysets);

  // check sets
  CPPUNIT_ASSERT_EQUAL( (size_t)2, SV->getContainer().size() );
  CPPUNIT_ASSERT_EQUAL( (size_t)1, SV->getContainer()[0]->size() );
  CPPUNIT_ASSERT_EQUAL( (size_t)2, SV->getContainer()[1]->size() );

}

/** Unit tests for countSetsTillLevel()
 *
 */
void IndexSetContainerTest::countSetsTillLevelTest()
{
  SV = new IndexSetContainer(allsets);

  CPPUNIT_ASSERT_EQUAL( (size_t)0, SV->countSetsTillLevel(0) );
  CPPUNIT_ASSERT_EQUAL( (size_t)2, SV->countSetsTillLevel(1) );
  CPPUNIT_ASSERT_EQUAL( (size_t)4, SV->countSetsTillLevel(2) );
  CPPUNIT_ASSERT_EQUAL( (size_t)5, SV->countSetsTillLevel(3) );
  CPPUNIT_ASSERT_EQUAL( (size_t)6, SV->countSetsTillLevel(4) );
  CPPUNIT_ASSERT_EQUAL( (size_t)6, SV->countSetsTillLevel(5) );
}
