/** \file periodentafel.cpp
 * 
 * Function implementations for the class periodentafel.
 * 
 */

using namespace std;

#include "periodentafel.hpp"

/************************************* Functions for class periodentafel ***************************/

/** constructor for class periodentafel
 * Initialises start and end of list and resets periodentafel::checkliste to false.
 */
periodentafel::periodentafel() 
{ 
  start = new element; 
  end = new element; 
  start->previous = NULL; 
  start->next = end; 
  end->previous = start; 
  end->next = NULL;
};

/** destructor for class periodentafel
 * Removes every element and afterwards deletes start and end of list.
 */
periodentafel::~periodentafel() 
{ 
  CleanupPeriodtable(); 
  delete(end); 
  delete(start); 
}; 

/** Adds element to period table list
 * \param *pointer element to be added
 * \return true - succeeded, false - does not occur
 */
bool periodentafel::AddElement(element *pointer) 
{ 
  pointer->sort = &pointer->Z;
  if (pointer->Z < 1 && pointer->Z >= MAX_ELEMENTS)
    cout << Verbose(0) << "Invalid Z number!\n";
  return add(pointer, end); 
};

/** Removes element from list.
 * \param *pointer element to be removed
 * \return true - succeeded, false - element not found
 */
bool periodentafel::RemoveElement(element *pointer) 
{ 
  return remove(pointer, start, end); 
};

/** Removes every element from the period table.
 * \return true - succeeded, false - does not occur
 */
bool periodentafel::CleanupPeriodtable() 
{ 
  return cleanup(start,end); 
};

/** Finds an element by its atomic number.
 * If element is not yet in list, datas are asked and stored in database.
 * \param Z atomic number
 * \return pointer to element
 */
element * periodentafel::FindElement(int Z)
{
  element *walker = find(&Z, start,end);
  if (walker == NULL) { // not found: enter and put into db
    cout << Verbose(0) << "Element not found in database, please enter." << endl;
    walker = new element;
    cout << Verbose(0) << "Mass: " << endl;
    cin >> walker->mass;
    walker->Z = Z;  
    cout << Verbose(0) << "Atomic number: " << walker->Z << endl;  
    cout << Verbose(0) << "Name [max 64 chars]: " << endl;
    cin >> walker->name;
    cout << Verbose(0) << "Short form [max 3 chars]: " << endl;
    cin >> walker->symbol;
    periodentafel::AddElement(walker);
  }
  return(walker);
};

/** Finds an element by its atomic number.
 * If element is not yet in list, datas are asked and stored in database.
 * \param shorthand chemical symbol of the element, e.g. H for hydrogene
 * \return pointer to element
 */
element * periodentafel::FindElement(char *shorthand) const
{
  element *walker =  periodentafel::start;
  while (walker->next != periodentafel::end) {
    walker = walker->next;
    if (strncmp(walker->symbol, shorthand, 3) == 0)
      return(walker);
  }
  return (NULL);
};

/** Asks for element number and returns pointer to element
 */
element * periodentafel::AskElement() 
{
  element *walker = NULL;
  int Z;
  do {
    cout << Verbose(0) << "Atomic number Z: ";
    cin >> Z;
    walker = this->FindElement(Z);  // give type
  } while (walker == NULL);
  return walker;
};


/** Prints period table to given stream.
 * \param output stream
 */ 
bool periodentafel::Output(ofstream *output) const
{
  bool result = true;
  element *walker = start;
  if (output != NULL) {
    while (walker->next != end) {
      walker = walker->next;
      result = result && walker->Output(output);
    }
    return result;
  } else 
    return false;
};

/** Prints period table to given stream.
 * \param *output output stream
 * \param *checkliste elements table for this molecule
 */ 
bool periodentafel::Checkout(ofstream *output, const int *checkliste) const
{
  element *walker = start;
  bool result = true;
  int No = 1;

  if (output != NULL) {
    *output << "# Ion type data (PP = PseudoPotential, Z = atomic number)" << endl;
    *output << "#Ion_TypeNr.\tAmount\tZ\tRGauss\tL_Max(PP)L_Loc(PP)IonMass\t# chemical name, symbol" << endl;
    while (walker->next != end) {
      walker = walker->next;
      if ((walker != NULL) && (walker->Z > 0) && (walker->Z < MAX_ELEMENTS) && (checkliste[walker->Z])) {
        walker->No = No;
        result = result && walker->Checkout(output, No++, checkliste[walker->Z]);      
      }
    }
    return result;
  } else 
    return false;
};


/** Loads element list from file.
 * \param *path to to standard file names
 */
bool periodentafel::LoadPeriodentafel(char *path)
{
  ifstream infile;
  double tmp;
  element *ptr;
  bool status = true;
  bool otherstatus = true;
  char *filename = new char[MAXSTRINGSIZE];
  
  // fill elements DB
  strncpy(filename, path, MAXSTRINGSIZE);
  strncat(filename, "/", MAXSTRINGSIZE-strlen(filename));
  strncat(filename, STANDARDELEMENTSDB, MAXSTRINGSIZE-strlen(filename));
  infile.open(filename);
  if (infile != NULL) {
    infile.getline(header1, MAXSTRINGSIZE);
    infile.getline(header2, MAXSTRINGSIZE); // skip first two header lines
    cout <<  "Parsed elements:";
    while (!infile.eof()) {
      element *neues = new element;
      infile >> neues->name;
      //infile >> ws;
      infile >> neues->symbol;
      //infile >> ws;
      infile >> neues->period;
      //infile >> ws;
      infile >> neues->group;
      //infile >> ws;
      infile >> neues->block;
      //infile >> ws;
      infile >> neues->Z;
      //infile >> ws;
      infile >> neues->mass;
      //infile >> ws;
      infile >> neues->CovalentRadius;
      //infile >> ws;
      infile >> neues->VanDerWaalsRadius;
      //infile >> ws;
      infile >> ws;
      cout << " " << neues->symbol;
      //neues->Output((ofstream *)&cout);
      if ((neues->Z > 0) && (neues->Z < MAX_ELEMENTS))
        periodentafel::AddElement(neues);
      else {
        cout << "Could not parse element: ";
        neues->Output((ofstream *)&cout);
      }
    }
    cout << endl;
    infile.close();
    infile.clear();
  } else
    status = false;

  // fill valence DB per element
  strncpy(filename, path, MAXSTRINGSIZE);
  strncat(filename, "/", MAXSTRINGSIZE-strlen(filename));
  strncat(filename, STANDARDVALENCEDB, MAXSTRINGSIZE-strlen(filename));
  infile.open(filename);
  if (infile != NULL) {
    while (!infile.eof()) {
    	infile >> tmp;
    	infile >> ws;
    	infile >> FindElement((int)tmp)->Valence;
    	infile >> ws;
    	//cout << Verbose(3) << "Element " << (int)tmp << " has " << FindElement((int)tmp)->Valence << " valence electrons." << endl;
    }
    infile.close();
    infile.clear();
  } else
    otherstatus = false;

  // fill valence DB per element
  strncpy(filename, path, MAXSTRINGSIZE);
  strncat(filename, "/", MAXSTRINGSIZE-strlen(filename));
  strncat(filename, STANDARDORBITALDB, MAXSTRINGSIZE-strlen(filename));
  infile.open(filename);
  if (infile != NULL) {
    while (!infile.eof()) {
      infile >> tmp;
      infile >> ws;
      infile >> FindElement((int)tmp)->NoValenceOrbitals;
      infile >> ws;
      //cout << Verbose(3) << "Element " << (int)tmp << " has " << FindElement((int)tmp)->NoValenceOrbitals << " number of singly occupied valence orbitals." << endl;
    }
    infile.close();
    infile.clear();
  } else
    otherstatus = false;
  
  // fill H-BondDistance DB per element
  strncpy(filename, path, MAXSTRINGSIZE);
  strncat(filename, "/", MAXSTRINGSIZE-strlen(filename));
  strncat(filename, STANDARDHBONDDISTANCEDB, MAXSTRINGSIZE-strlen(filename));
  infile.open(filename);
  if (infile != NULL) {
    while (!infile.eof()) {
    	infile >> tmp;
      ptr = FindElement((int)tmp);
    	infile >> ws;
      infile >> ptr->HBondDistance[0];
      infile >> ptr->HBondDistance[1];
      infile >> ptr->HBondDistance[2];
    	infile >> ws;
      //cout << Verbose(3) << "Element " << (int)tmp << " has " << FindElement((int)tmp)->HBondDistance[0] << " Angstrom typical distance to hydrogen." << endl;
    }
    infile.close();
    infile.clear();
  } else
    otherstatus = false;
  
  // fill H-BondAngle DB per element
  strncpy(filename, path, MAXSTRINGSIZE);
  strncat(filename, "/", MAXSTRINGSIZE-strlen(filename));
  strncat(filename, STANDARDHBONDANGLEDB, MAXSTRINGSIZE-strlen(filename));
  infile.open(filename);
  if (infile != NULL) {
    while (!infile.eof()) {
      infile >> tmp;
      ptr = FindElement((int)tmp);
      infile >> ws;
      infile >> ptr->HBondAngle[0];
      infile >> ptr->HBondAngle[1];
      infile >> ptr->HBondAngle[2];
      infile >> ws;
      //cout << Verbose(3) << "Element " << (int)tmp << " has " << FindElement((int)tmp)->HBondAngle[0] << ", " << FindElement((int)tmp)->HBondAngle[1] << ", " << FindElement((int)tmp)->HBondAngle[2] << " degrees bond angle for one, two, three connected hydrogens." << endl;
    }
    infile.close();
  } else
    otherstatus = false;
  
  if (!otherstatus)
    cerr << "WARNING: Something went wrong while parsing the other databases!" << endl;
  
  return status;
};

/** Stores element list to file.
 */
bool periodentafel::StorePeriodentafel(char *path) const
{
  bool result = true;
  ofstream f;
  char filename[MAXSTRINGSIZE];
  
  strncpy(filename, path, MAXSTRINGSIZE);
  strncat(filename, "/", MAXSTRINGSIZE-strlen(filename));
  strncat(filename, STANDARDELEMENTSDB, MAXSTRINGSIZE-strlen(filename));
  f.open(filename);
  if (f != NULL) {
    f << header1 << endl;
    f << header2 << endl;
    element *walker = periodentafel::start;
    while (walker->next != periodentafel::end) {
      walker = walker->next;
      result = result && walker->Output(&f);
    }
    f.close();
  } else
    result = false;
  return result;
};
