/*
 * tesselation.hpp
 *
 *  The tesselation class is meant to contain the envelope (concave, convex or neither) of a set of Vectors.
 *  As we actually mean this stuff for atoms, we have to encapsulate it all a bit.
 *
 *  Created on: Aug 3, 2009
 *      Author: heber
 */

#ifndef TESSELATION_HPP_
#define TESSELATION_HPP_

using namespace std;

/*********************************************** includes ***********************************/

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <map>
#include <list>
#include <set>
#include <stack>

#include "BoundaryMaps.hpp"
#include "PointCloud.hpp"
#include "TesselPoint.hpp"
#include "atom_particleinfo.hpp"
#include "Helpers/helpers.hpp"
#include "LinearAlgebra/Vector.hpp"


/****************************************** forward declarations *****************************/

class BoundaryPointSet;
class BoundaryLineSet;
class BoundaryTriangleSet;
class CandidateForTesselation;
class LinkedCell;
class Tesselation;
class Plane;

/********************************************** definitions *********************************/

enum { DoTecplotOutput=1 };
enum { DoRaster3DOutput=1 };
enum { DoVRMLOutput=0 };

extern "C" const char *TecplotSuffix;
extern "C" const char *Raster3DSuffix;
extern "C" const char *VRMLSUffix;

extern "C" const double ParallelEpsilon;

// ======================================================= some template functions =========================================

/********************************************** declarations *******************************/

// =========================================================== class TESSELATION ===========================================

/** Contains the envelope to a PointCloud.
 */
class Tesselation : public PointCloud {
  public:

    Tesselation();
    virtual ~Tesselation();

    void AddTesselationPoint(TesselPoint* Candidate, const int n);
    void SetTesselationPoint(TesselPoint* Candidate, const int n) const;
    void AddTesselationLine(const Vector * OptCenter, const BoundaryPointSet * const candidate, class BoundaryPointSet *a, class BoundaryPointSet *b, const int n);
    void AddNewTesselationTriangleLine(class BoundaryPointSet *a, class BoundaryPointSet *b, const int n);
    void AddExistingTesselationTriangleLine(class BoundaryLineSet *FindLine, int n);
    void AddTesselationTriangle();
    void AddTesselationTriangle(const int nr);
    void AddCandidateTriangle(CandidateForTesselation &CandidateLine, enum centers type);
    void AddDegeneratedTriangle(CandidateForTesselation &CandidateLine, const double RADIUS, const LinkedCell *LC);
    void AddCandidatePolygon(CandidateForTesselation CandidateLine, const double RADIUS, const LinkedCell *LC);
    void RemoveTesselationTriangle(class BoundaryTriangleSet *triangle);
    void RemoveTesselationLine(class BoundaryLineSet *line);
    void RemoveTesselationPoint(class BoundaryPointSet *point);
    bool CheckDegeneracy(CandidateForTesselation &CandidateLine, const double RADIUS, const LinkedCell *LC) const;


    // concave envelope
    bool FindStartingTriangle(const double RADIUS, const LinkedCell *LC);
    void FindSecondPointForTesselation(class TesselPoint* a, Vector Oben, class TesselPoint*& OptCandidate, double Storage[3], double RADIUS, const LinkedCell *LC);
    void FindThirdPointForTesselation(const Vector &NormalVector, const Vector &SearchDirection, const Vector &OldSphereCenter, CandidateForTesselation &CandidateLine, const class BoundaryPointSet  * const ThirdNode, const double RADIUS, const LinkedCell *LC) const;
    bool FindNextSuitableTriangle(CandidateForTesselation &CandidateLine, const BoundaryTriangleSet &T, const double& RADIUS, const LinkedCell *LC);
    bool FindCandidatesforOpenLines(const double RADIUS, const LinkedCell *&LCList);
    int CheckPresenceOfTriangle(class TesselPoint *Candidates[3]) const;
    class BoundaryTriangleSet * GetPresentTriangle(TesselPoint *Candidates[3]);

    // convex envelope
    void TesselateOnBoundary(const PointCloud * const cloud);
    void GuessStartingTriangle();
    bool InsertStraddlingPoints(const PointCloud *cloud, const LinkedCell *LC);
    double RemovePointFromTesselatedSurface(class BoundaryPointSet *point);
    class BoundaryLineSet * FlipBaseline(class BoundaryLineSet *Base);
    double PickFarthestofTwoBaselines(class BoundaryLineSet *Base);
    class BoundaryPointSet *IsConvexRectangle(class BoundaryLineSet *Base);
    IndexToIndex * FindAllDegeneratedTriangles();
    IndexToIndex * FindAllDegeneratedLines();
    void RemoveDegeneratedTriangles();
    void AddBoundaryPointByDegeneratedTriangle(class TesselPoint *point, LinkedCell *LC);
    int CorrectAllDegeneratedPolygons();

    TesselPointSet * GetAllConnectedPoints(const TesselPoint* const Point) const;
    TriangleSet * GetAllTriangles(const BoundaryPointSet * const Point) const;
    ListOfTesselPointList * GetPathsOfConnectedPoints(const TesselPoint* const Point) const;
    ListOfTesselPointList * GetClosedPathsOfConnectedPoints(const TesselPoint* const Point) const;
    TesselPointList * GetCircleOfSetOfPoints(TesselPointSet *SetOfNeighbours, const TesselPoint* const Point, const Vector &Reference) const;
    TesselPointList * GetCircleOfConnectedTriangles(TesselPointSet *SetOfNeighbours, const TesselPoint* const Point, const Vector &Reference) const;
    class BoundaryPointSet * GetCommonEndpoint(const BoundaryLineSet * line1, const BoundaryLineSet * line2) const;
    TriangleList * FindTriangles(const TesselPoint* const Points[3]) const;
    TriangleList * FindClosestTrianglesToVector(const Vector &x, const LinkedCell* LC) const;
    BoundaryTriangleSet * FindClosestTriangleToVector(const Vector &x, const LinkedCell* LC) const;
    bool IsInnerPoint(const Vector &Point, const LinkedCell* const LC) const;
    double GetDistanceSquaredToTriangle(const Vector &Point, const BoundaryTriangleSet* const triangle) const;
    double GetDistanceToSurface(const Vector &Point, const LinkedCell* const LC) const;
    BoundaryTriangleSet * GetClosestTriangleOnSurface(const Vector &Point, const LinkedCell* const LC) const;
    bool AddBoundaryPoint(TesselPoint * Walker, const int n);
    DistanceToPointMap * FindClosestBoundaryPointsToVector(const Vector &x, const LinkedCell* LC) const;
    BoundaryLineSet * FindClosestBoundaryLineToVector(const Vector &x, const LinkedCell* LC) const;

    // print for debugging
    void PrintAllBoundaryPoints(ofstream *out) const;
    void PrintAllBoundaryLines(ofstream *out) const;
    void PrintAllBoundaryTriangles(ofstream *out) const;

    // store envelope in file
    void Output(const char *filename, const PointCloud * const cloud);

    PointMap PointsOnBoundary;
    LineMap LinesOnBoundary;
    CandidateMap OpenLines;
    TriangleMap TrianglesOnBoundary;
    int PointsOnBoundaryCount;
    int LinesOnBoundaryCount;
    int TrianglesOnBoundaryCount;

    // PointCloud implementation for PointsOnBoundary
    virtual Vector *GetCenter(ofstream *out) const;
    virtual TesselPoint *GetPoint() const;
    virtual void GoToNext() const;
    virtual void GoToFirst() const;
    virtual bool IsEmpty() const;
    virtual bool IsEnd() const;

    class BoundaryPointSet *BPS[2];
    class BoundaryLineSet *BLS[3];
    class BoundaryTriangleSet *BTS;
    class BoundaryTriangleSet *LastTriangle;
    int TriangleFilesWritten;

  private:
    static const double HULLEPSILON; //!< TODO: Get rid of HULLEPSILON, points to numerical instabilities

    mutable class BoundaryPointSet *TPS[3]; //this is a Storage for pointers to triangle points, this and BPS[2] needed due to AddLine restrictions

    mutable PointMap::const_iterator InternalPointer;

    //bool HasOtherBaselineBetterCandidate(const BoundaryLineSet * const BaseRay, const TesselPoint * const OptCandidate, double ShortestAngle, double RADIUS, const LinkedCell * const LC) const;
    void FindDegeneratedCandidatesforOpenLines(TesselPoint * const Sprinter, const Vector * const OptCenter);
};


#endif /* TESSELATION_HPP_ */
