/*
 * DiscreteValues_impl.hpp
 *
 *  Created on: Sep 28, 2011
 *      Author: heber
 */

#ifndef DISCRETEVALUE_IMPL_HPP_
#define DISCRETEVALUE_IMPL_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <algorithm>
#include <vector>

#include <boost/any.hpp>

#include "CodePatterns/Assert.hpp"

#include "CodePatterns/Log.hpp"

// static member
template <class T> ConvertTo<T> DiscreteValue<T>::Converter;

/** Constructor of class DiscreteValue.
 */
template <class T>
DiscreteValue<T>::DiscreteValue() :
  ValueSet(false)
{}

/** Constructor of class DiscreteValue with set of valid values.
 *
 * @param _ValidValues vector with all valid values
 */
template <class T>
DiscreteValue<T>::DiscreteValue(const std::vector<T> &_ValidValues) :
  ValueSet(false),
  ValidValues(_ValidValues)
{}

/** Destructor of class DiscreteValue.
 */
template <class T>
DiscreteValue<T>::~DiscreteValue()
{}

/** Checks whether \a _value is a valid value.
 * \param _value value to check for validity.
 * \return true - \a _value is valid, false - is not
 */
template <class T>
bool DiscreteValue<T>::isValid(const std::string _value) const
{
  const T castvalue = Converter(_value);
  return isValidValue(castvalue);
}

/** Getter of value, returning string.
 *
 * @return string value
 */
template <class T>
const std::string DiscreteValue<T>::get() const
{
  ASSERT(ValueSet,
      "DiscreteValue<T>::get() - requesting unset value.");
  return toString(getValue());
}

/** Setter of value for string
 *
 * @param _value string containing new value
 */
template <class T>
void DiscreteValue<T>::set(const std::string _value)
{
  const T castvalue = Converter(_value);
  setValue(castvalue);
}


/** Internal function for finding the index of a desired value.
 *
 * \note As this is internal, we do not ASSERT value's presence, but return -1
 * such that other functions may ASSERT on that.
 *
 * \param _value value to get the index of
 * \return index such that ValidValues[index] == _value
 */
template <class T>
const size_t DiscreteValue<T>::findIndexOfValue(const T &_value) const
{
  size_t index = 0;
  const size_t max = ValidValues.size();
  for (; index < max; ++index) {
    if (ValidValues[index] == _value)
      break;
  }
  if (index == max)
    return (size_t)-1;
  else
    return index;
}

/** Adds another value to the valid ones.
 *
 * We check whether its already present, otherwise we throw an Assert::AssertionFailure.
 *
 * @param _value value to add
 */
template <class T>
void DiscreteValue<T>::appendValidValue(const T &_value)
{
  ASSERT(!isValidValue(_value),
      "DiscreteValue<>::appendValidValue() - value "+toString(_value)+" is already among the valid");
  ValidValues.push_back(_value);
}

/** Returns all possible valid values.
 *
 * @return vector with all allowed values
 */
template <class T>
const std::vector<T> &DiscreteValue<T>::getValidValues() const
{
  return ValidValues;
}

/** Sets the value.
 *
 * We check for its validity, otherwise we throw an Assert::AssertionFailure.
 *
 * @param _value const reference of value to set
 */
template <class T>
void DiscreteValue<T>::setValue(const T &_value)
{
  const size_t index = findIndexOfValue(_value);
  ASSERT(index != (size_t)-1,
      "DiscreteValue<>::set() - value "+toString(_value)+" is not valid.");
  if (!ValueSet)
    ValueSet = true;
  value = index;
}

/** Getter for the set value.
 *
 * We check whether it has been set, otherwise we throw an Assert::AssertionFailure.
 *
 * @return set value
 */
template <class T>
const T & DiscreteValue<T>::getValue() const
{
  ASSERT(ValueSet,
      "DiscreteValue<>::get() - value has never been set.");
  return ValidValues[value];
}

/** Checks whether \a _value is a valid value.
 * \param _value value to check for validity.
 * \return true - \a _value is valid, false - is not
 */
template <class T>
bool DiscreteValue<T>::isValidValue(const T &_value) const
{
  typename ValidRange::const_iterator iter = std::find(ValidValues.begin(), ValidValues.end(), _value);
  if (iter != ValidValues.end()) {
    //std::cout << "Found " << _value << ":" << *iter << std::endl;
    return true;
  } else {
    //std::cout << "Did not find " << _value << "." << std::endl;
    return false;
  }
}

#endif /* DISCRETEVALUE_IMPL_HPP_ */
