/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * CountBondsUnitTest.cpp
 *
 *  Created on: Mar 30, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

using namespace std;

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include <iostream>
#include <stdio.h>
#include <cstring>

#include "CodePatterns/Assert.hpp"

#include "Analysis/analysis_bonds.hpp"
#include "Atom/atom.hpp"
#include "Bond/bond.hpp"
#include "CodePatterns/Log.hpp"
#include "Element/element.hpp"
#include "Element/periodentafel.hpp"
#include "Graph/BondGraph.hpp"
#include "molecule.hpp"
#include "MoleculeListClass.hpp"
#include "World.hpp"

#include "CountBondsUnitTest.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( CountBondsTest );


void CountBondsTest::setUp()
{
  atom *Walker = NULL;

  // construct element
  hydrogen = World::getInstance().getPeriode()->FindElement(1);
  oxygen = World::getInstance().getPeriode()->FindElement(8);
  CPPUNIT_ASSERT(hydrogen != NULL && "could not find element hydrogen");
  CPPUNIT_ASSERT(oxygen != NULL && "could not find element oxygen");

  //setVerbosity(3);

  // construct molecule (water molecule)
  TestMolecule1 = World::getInstance().createMolecule();
  CPPUNIT_ASSERT(TestMolecule1 != NULL && "could not create second molecule");
  Walker = World::getInstance().createAtom();
  CPPUNIT_ASSERT(Walker != NULL && "could not create atom");
  Walker->setType(hydrogen);
  Walker->setPosition(Vector(-0.2418, 0.9350, 0. ));
  TestMolecule1->AddAtom(Walker);
  Walker = World::getInstance().createAtom();
  CPPUNIT_ASSERT(Walker != NULL && "could not create atom");
  Walker->setType(hydrogen);
  Walker->setPosition(Vector(0.9658, 0., 0. ));
  TestMolecule1->AddAtom(Walker);
  Walker = World::getInstance().createAtom();
  CPPUNIT_ASSERT(Walker != NULL && "could not create atom");
  Walker->setType(oxygen);
  Walker->setPosition(Vector(0., 0., 0. ));
  TestMolecule1->AddAtom(Walker);

  molecules = World::getInstance().getMolecules();
  CPPUNIT_ASSERT(molecules != NULL && "could not obtain list of molecules");
  molecules->insert(TestMolecule1);

  // check that TestMolecule1 was correctly constructed
  CPPUNIT_ASSERT_EQUAL( TestMolecule1->getAtomCount(), 3 );

  // create a small file with table
  BG = new BondGraph(true);
  CPPUNIT_ASSERT(BG != NULL && "could not create BondGraph");

  // construct bond graphs
  World::AtomComposite Set1 = TestMolecule1->getAtomSet();
  BG->CreateAdjacency(Set1);
  CPPUNIT_ASSERT( TestMolecule1->getBondCount() != 0);
//  TestMolecule1->Output((ofstream *)&cout);
//  TestMolecule1->OutputBondsList(std::cout);
};


void CountBondsTest::tearDown()
{
  // remove the file
  delete(BG);

  World::purgeInstance();
};

/** UnitTest for CountBondsTest::BondsOfTwoTest().
 */
void CountBondsTest::BondsOfTwoTest()
{
  CPPUNIT_ASSERT_EQUAL( 2 , CountBondsOfTwo(molecules, hydrogen, oxygen) );
  CPPUNIT_ASSERT_EQUAL( 0 , CountBondsOfTwo(molecules, hydrogen, hydrogen) );
  CPPUNIT_ASSERT_EQUAL( 0 , CountBondsOfTwo(molecules, oxygen, oxygen) );
};

/** UnitTest for CountBondsTest::BondsOfThreeTest().
 */
void CountBondsTest::BondsOfThreeTest()
{
  CPPUNIT_ASSERT_EQUAL( 1 , CountBondsOfThree(molecules, hydrogen, oxygen, hydrogen) );
  CPPUNIT_ASSERT_EQUAL( 0 , CountBondsOfThree(molecules, oxygen, hydrogen, oxygen) );
};

/** UnitTest for CountBondsTest::HydrogenBridgeBondsTest().
 */
void CountBondsTest::HydrogenBridgeBondsTest()
{
  double mirror[3];
  CPPUNIT_ASSERT(mirror != NULL && "could not create array of doubles");
  for (int i=0;i<3;i++)
    mirror[i] = -1.;
  Vector Translator;

  // create TestMolecule2 as copy
  TestMolecule2 = TestMolecule1->CopyMolecule();
  molecules->insert(TestMolecule2);
  CPPUNIT_ASSERT_EQUAL( TestMolecule2->getAtomCount(), 3 );

  cout << "Case 1: offset of (3,0,0), hence angles are (104.5, 0, 75.5, 180) < 30." << endl;
  Translator  = Vector(3,0,0);
  TestMolecule1->Translate(Translator);
  CPPUNIT_ASSERT_EQUAL( 1 , CountHydrogenBridgeBonds(molecules, NULL, NULL) );
  CPPUNIT_ASSERT_EQUAL( 0 , CountHydrogenBridgeBonds(molecules, oxygen, NULL) );
  Translator = Vector(-3,0,0);
  TestMolecule1->Translate(Translator);

  cout << "Case 2: offset of (0,3,0), hence angle are (14.5, 165.5, 90) < 30 (only three, because other 90 is missing due to first H01 only fulfilling H-bond criteria)." << endl;
  Translator = Vector(0,3,0);
  TestMolecule1->Translate(Translator);
  CPPUNIT_ASSERT_EQUAL( 1 , CountHydrogenBridgeBonds(molecules, NULL, NULL) );
  Translator = Vector(0,-3,0);
  TestMolecule1->Translate(Translator);

  cout << "Case 3: offset of (0,-3,0) and mirror, hence angle are (165.5, 90, 165.5, 90) > 30." << endl;
  Translator = Vector(0,-3,0);
  TestMolecule1->Scale(&mirror[0]);
  TestMolecule1->Translate(Translator);
  CPPUNIT_ASSERT_EQUAL( 0 , CountHydrogenBridgeBonds(molecules, NULL, NULL) );
  Translator = Vector(0,3,0);
  TestMolecule1->Translate(Translator);
  TestMolecule1->Scale(&mirror[0]);

  cout << "Case 4: offset of (2,1,0), hence angle are (78, 26.6, 102, 153.4) < 30." << endl;
  Translator = Vector(2,1,0);
  TestMolecule1->Translate(Translator);
  CPPUNIT_ASSERT_EQUAL( 1 , CountHydrogenBridgeBonds(molecules, NULL, NULL) );
  Translator = Vector(-2,-1,0);
  TestMolecule1->Translate(Translator);

  cout << "Case 5: offset of (0,0,3), hence angle are (90, 90, 90, 90) > 30." << endl;
  Translator = Vector(0,0,3);
  TestMolecule1->Translate(Translator);
  CPPUNIT_ASSERT_EQUAL( 0 , CountHydrogenBridgeBonds(molecules, NULL, NULL) );
  Translator = Vector(0,0,-3);
  TestMolecule1->Translate(Translator);

  cout << "Case 6: offset of (-3,0,0) and mirror, hence angle are (75.5, 180, 104.5, 180) > 30." << endl;
  Translator = Vector(-3,0,0);
  TestMolecule1->Scale(&mirror[0]);
  TestMolecule1->Translate(Translator);
  CPPUNIT_ASSERT_EQUAL( 0 , CountHydrogenBridgeBonds(molecules, NULL, NULL) );
  Translator = Vector(3,0,0);
  TestMolecule1->Translate(Translator);
  TestMolecule1->Scale(&mirror[0]);

  cout << "Case 7: offset of (3,0,0) and mirror, hence angles are (104.5, 0, 104.5, 0) < 30, but interfering hydrogens." << endl;
  Translator = Vector(3,0,0);
  TestMolecule1->Scale(&mirror[0]);
  TestMolecule1->Translate(Translator);
  CPPUNIT_ASSERT_EQUAL( 0 , CountHydrogenBridgeBonds(molecules, NULL, NULL) );
  Translator = Vector(-3,0,0);
  TestMolecule1->Translate(Translator);
  TestMolecule1->Scale(&mirror[0]);

  cout << "Case 8: offset of (0,3,0), hence angle are (14.5, 90, 14.5, 90) < 30, but interfering hydrogens." << endl;
  Translator = Vector(0,3,0);
  TestMolecule1->Scale(&mirror[0]);
  TestMolecule1->Translate(Translator);
  CPPUNIT_ASSERT_EQUAL( 0 , CountHydrogenBridgeBonds(molecules, NULL, NULL) );
  Translator = Vector(0,-3,0);
  TestMolecule1->Translate(Translator);
  TestMolecule1->Scale(&mirror[0]);
};
