/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * AtomIdSetUnitTest.cpp
 *
 *  Created on: Feb 21, 2012
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include <algorithm>

#include "Atom/atom.hpp"
#include "AtomIdSet.hpp"
#include "CodePatterns/Assert.hpp"
#include "World.hpp"

#include "AtomIdSetUnitTest.hpp"


#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( AtomIdSetUnittest );

size_t AtomIdSetUnittest::MaxAtoms = 6;

void AtomIdSetUnittest::setUp(){
  // failing asserts should be thrown
  ASSERT_DO(Assert::Throw);

  atomVector.resize((size_t)MaxAtoms);
  std::generate_n(atomVector.begin(), MaxAtoms, boost::bind(
    static_cast<atom* (World::*)(World::CreateAtomNotificationType)>(&World::createAtom), 
    World::getPointer(), 
    World::CreateAtomNotificationType::NOTIFY_ON_CREATE_ATOM)
  );
}

void AtomIdSetUnittest::tearDown()
{
  World::purgeInstance();
}

void AtomIdSetUnittest::inserteraseTest()
{
  // insert all
  for (size_t i=0; i< MaxAtoms; ++i) {
    atoms.insert(atomVector[i]);
  }
  CPPUNIT_ASSERT_EQUAL( (size_t)MaxAtoms, atoms.size() );

  // erase them again
  for (size_t i=0; i< MaxAtoms; ++i) {
    atoms.erase(atomVector[i]);
  }
  CPPUNIT_ASSERT_EQUAL( (size_t)0, atoms.size() );

  {
    // use atomVector in cstor
    AtomIdSet otherAtoms(atomVector);
    CPPUNIT_ASSERT_EQUAL( (size_t)MaxAtoms, otherAtoms.size() );
  }
}

void AtomIdSetUnittest::positionTest()
{
  // non-const iterators
  CPPUNIT_ASSERT( atoms.begin() == atoms.end() );

  // const iterators
  {
    AtomIdSet::const_iterator beg_iter = const_cast<const AtomIdSet &>(atoms).begin();
    AtomIdSet::const_iterator end_iter = const_cast<const AtomIdSet &>(atoms).end();
    CPPUNIT_ASSERT( beg_iter == end_iter );
  }

  // insert an element
  atoms.insert(atomVector[0]);
  CPPUNIT_ASSERT( atoms.begin() != atoms.end() );

  {
    AtomIdSet::const_iterator beg_iter = const_cast<const AtomIdSet &>(atoms).begin();
    AtomIdSet::const_iterator end_iter = const_cast<const AtomIdSet &>(atoms).end();
    CPPUNIT_ASSERT( beg_iter != end_iter );
    ++beg_iter;
    CPPUNIT_ASSERT( beg_iter == end_iter );
  }
}

void AtomIdSetUnittest::containsTest()
{
  // search for atom
  atoms.insert(atomVector[0]);
  CPPUNIT_ASSERT( atoms.contains(atomVector[0]) );
  atoms.erase(atomVector[0]);
  CPPUNIT_ASSERT( !atoms.contains(atomVector[0]) );

  // search for key
  atoms.insert(atomVector[0]->getId());
  CPPUNIT_ASSERT( atoms.contains(atomVector[0]->getId()) );
  atoms.erase(atomVector[0]->getId());
  CPPUNIT_ASSERT( !atoms.contains(atomVector[0]->getId()) );
}

void AtomIdSetUnittest::findTest()
{
  // search for atom
  atoms.insert(atomVector[0]);
  CPPUNIT_ASSERT( atoms.find(atomVector[0]) == const_cast<const AtomIdSet &>(atoms).begin() );
  CPPUNIT_ASSERT( atoms.find(atomVector[0]) != const_cast<const AtomIdSet &>(atoms).end() );
  atoms.erase(atomVector[0]);

  // search for key
  atoms.insert(atomVector[0]->getId());
  CPPUNIT_ASSERT( atoms.find(atomVector[0]->getId()) == const_cast<const AtomIdSet &>(atoms).begin() );
  CPPUNIT_ASSERT( atoms.find(atomVector[0]->getId()) != const_cast<const AtomIdSet &>(atoms).end() );
  atoms.erase(atomVector[0]->getId());
}


void AtomIdSetUnittest::emptyTest()
{
  // initially empty
  CPPUNIT_ASSERT( atoms.empty() );

  // filled with one then no more
  atoms.insert(atomVector[0]);
  CPPUNIT_ASSERT( !atoms.empty() );

  // erase and again empty
  atoms.erase(atomVector[0]);
  CPPUNIT_ASSERT( atoms.empty() );
}

void AtomIdSetUnittest::sizeTest()
{
  // initially empty
  CPPUNIT_ASSERT_EQUAL( (size_t)0, atoms.size() );

  // filled with one, then no more
  atoms.insert(atomVector[0]);
  CPPUNIT_ASSERT_EQUAL( (size_t)1, atoms.size() );

  // erase and again empty
  atoms.erase(atomVector[0]);
  CPPUNIT_ASSERT_EQUAL( (size_t)0, atoms.size() );
}

