/*
 * UndoRedoHelpers.hpp
 *
 *  Created on: Apr 5, 2012
 *      Author: heber
 */

#ifndef UNDOREDOHELPERS_HPP_
#define UNDOREDOHELPERS_HPP_


// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <vector>

#include "Atom/AtomicInfo.hpp"
#include "Bond/BondInfo.hpp"

namespace MoleCuilder {

  /** Adds removed atoms back to the world whose state is stored as AtomicInfo.
   *
   * @param atoms vector of atomicInfo
   * @return restoral was successful, at least atom could not be restored.
   */
  bool AddAtomsFromAtomicInfo(std::vector<AtomicInfo> &atoms);

  /** Removes atoms whose state information is stored as AtomicInfo.
   *
   * @param atoms vector of atomicInfo
   */
  void RemoveAtomsFromAtomicInfo(std::vector<AtomicInfo> &atoms);

  /** Stores the required bond information in for all \a atoms in \a bonds.
   *
   * @param atoms atoms whose bonds to store
   * @param bonds vector with bond information on return
   */
  void StoreBondInformationFromAtoms(
      const std::vector<const atom*> &atoms,
      std::vector<BondInfo> &bonds);

  /** Recreates bonds from information stored in \a bonds.
   *
   * @param bonds bond state information
   * @return true - all bonds restored, false - at least one bond could not be restored
   */
  bool AddBondsFromBondInfo(const std::vector< BondInfo > &bonds);

  /** Sets atoms to state information stored as AtomicInfo.
   *
   * @param movedatoms vector of atomicInfo
   */
  void SetAtomsFromAtomicInfo(const std::vector<AtomicInfo> &_movedatoms);

  /** Selects all atoms inside the given vector
   *
   * @param movedatoms vector of atomicInfo
   */
  void SelectAtomsFromAtomicInfo(const std::vector<AtomicInfo> &_movedatoms);

  /** Helper function to allow setting arbitrary atom vectorial information via some
   * \a setter function.
   *
   * @param movedatoms atoms whose info to change
   * @param MovedToVector vector with old vectorial information
   */
  void ResetByFunction(
      const std::vector<AtomicInfo> &movedatoms,
      const std::vector<Vector> &MovedToVector,
      boost::function<void(atom *, const Vector&)> &setter);

  /** Sets the atoms whose id is stored in given AtomicInfo in \a movedatoms
   *  to position in \a MovedToVector.
   *
   * @param movedatoms atoms whose position to change
   * @param MovedToVector vector with old positions
   */
  void ResetAtomPosition(const std::vector<AtomicInfo> &movedatoms, const std::vector<Vector> &MovedToVector);

  /** Sets the atoms whose id is stored in given AtomicInfo in \a movedatoms
   *  to position in \a MovedToVector.
   *
   * @param movedatoms atoms whose position to change
   * @param VelocityVector vector with old velocities
   */
  void ResetAtomVelocity(const std::vector<AtomicInfo> &movedatoms, const std::vector<Vector> &VelocityVector);

  /** Sets the atoms whose id is stored in given AtomicInfo in \a movedatoms
   *  to position in \a MovedToVector.
   *
   * @param movedatoms atoms whose position to change
   * @param ForceVector vector with old forces
   */
  void ResetAtomForce(const std::vector<AtomicInfo> &movedatoms, const std::vector<Vector> &ForceVector);

  /** Remove all molecules identified by their ids given in \a ids.
   *
   * @param ids vector of molecular ids to remove
   */
  void RemoveMoleculesWithAtomsByIds(const std::vector<moleculeId_t> &ids);

  /** Removes the last time step for all \a movedatoms
   *
   * @param movedatoms atoms whose last step in time to remove
   */
  void removeLastStep(const std::vector<atomId_t> &movedatoms);

  /** Adds another time step to all \a movedatoms.
   *
   * Note that the time step is initialized to zero.
   *
   * @param _ids atoms whose last step in time to remove
   */
  void addNewStep(const std::vector<atomId_t> &_ids);

  /** Adds another time step to all \a movedatoms.
   *
   * Note that the time step is initialized to zero. This gives you a chance
   * to set the time step and call setAtomsFromAtomicInfo() yourself.
   *
   * @param _movedatoms atoms whose last step in time to remove
   */
  void addNewStep(const std::vector<AtomicInfo> &_movedatoms);

  /** Helper function to extract id information from vector of AtomicInfo.
   *
   * @param movedatoms atoms whose ids to extract
   */
  std::vector<atomId_t> getIdsFromAtomicInfo(const std::vector<AtomicInfo> &movedatoms);
}



#endif /* UNDOREDOHELPERS_HPP_ */
