/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * Copyright (C)  2013 Frederik Heber. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * GLMoleculeObject_bond.cpp
 *
 *  Created on: Aug 17, 2011
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "GLMoleculeObject_bond.hpp"

#include <Qt3D/qglmaterial.h>
#include <Qt3D/qglscenenode.h>

#include "CodePatterns/MemDebug.hpp"


#include <cmath>

#include "CodePatterns/Assert.hpp"
#include "CodePatterns/Log.hpp"
#include "CodePatterns/Observer/Notification.hpp"
#include "CodePatterns/Observer/ObserverLog.hpp"
#include "Descriptors/AtomIdDescriptor.hpp"
#include "Atom/atom.hpp"
#include "Bond/bond.hpp"
#include "Element/element.hpp"
#include "Helpers/defs.hpp"
#include "LinearAlgebra/Line.hpp"
#include "LinearAlgebra/Vector.hpp"
#include "World.hpp"


// static entities
const Observable::channels_t
GLMoleculeObject_bond::BondPositionChannels(1, AtomObservable::PositionChanged);
const Observable::channels_t
GLMoleculeObject_bond::BondDegreeChannels(1, BondObservable::DegreeChanged);
const Observable::channels_t
GLMoleculeObject_bond::BondElementChannels(1, AtomObservable::ElementChanged);

GLMoleculeObject_bond::GLMoleculeObject_bond(
    QGLSceneNode *mesh[],
    QObject *parent,
    const bondIds_t bondIds,
    const enum SideOfBond side) :
  GLMoleculeObject(mesh, parent),
  Observer(std::string("GLMoleculeObject_bond")
      +toString(bondIds.first)
      +std::string("-")
      +toString(bondIds.second)),
  leftatomId(bondIds.first),
  rightatomId(bondIds.second),
  leftowner(getAtom(leftatomId)),
  rightowner(getAtom(rightatomId)),
  bondowner(getAtom(leftatomId)->getBond(getAtom(rightatomId)).get()),
  BondSide(side),
  leftPosition(
      leftowner,
      boost::bind(&GLMoleculeObject_bond::updateLeftPosition, this),
      "BondleftPosition_"+toString(leftatomId),
      updateLeftPosition(),
      BondPositionChannels),
  rightPosition(
      rightowner,
      boost::bind(&GLMoleculeObject_bond::updateRightPosition, this),
      "BondrightPosition_"+toString(rightatomId),
      updateRightPosition(),
      BondPositionChannels),
  leftElement(
      leftowner,
      boost::bind(&GLMoleculeObject_bond::updateLeftElement, this),
      "BondleftElement"+toString(leftatomId),
      updateLeftElement(),
      BondElementChannels),
  rightElement(
      rightowner,
      boost::bind(&GLMoleculeObject_bond::updateRightElement, this),
      "BondrightElement"+toString(rightatomId),
      updateRightElement(),
      BondElementChannels),
  Degree(
      bondowner,
      boost::bind(&GLMoleculeObject_bond::updateDegree, this),
      "BondDegree"+toString(leftatomId)+"_"+toString(rightatomId),
      updateDegree(),
      BondDegreeChannels),
  leftobservable_enabled(false),
  rightobservable_enabled(false)
{
  // sign on as observer (obtain non-const instance before)
  bondowner->signOn(this, BondObservable::BondRemoved);
  bondowner->signOn(this, BondObservable::DegreeChanged);
  bond_enabled = true;
  leftowner->signOn(this, AtomObservable::PositionChanged);
  leftowner->signOn(this, AtomObservable::ElementChanged);
  leftobservable_enabled = true;
  rightowner->signOn(this, AtomObservable::PositionChanged);
  rightowner->signOn(this, AtomObservable::ElementChanged);
  rightobservable_enabled = true;

  resetElement();
  resetPosition();
  resetWidth();

  connect(this, SIGNAL(elementChanged()), this, SLOT(resetElement()), Qt::QueuedConnection);
  connect(this, SIGNAL(positionChanged()), this, SLOT(resetPosition()), Qt::QueuedConnection);
  connect(this, SIGNAL(degreeChanged()), this, SLOT(resetWidth()), Qt::QueuedConnection);
}

GLMoleculeObject_bond::~GLMoleculeObject_bond()
{
  LOG(3, "DEBUG: Destroying  GLMoleculeObject_bond to bond [" <<
      leftatomId << "," << rightatomId << "] and side " << BondSide << ".");
  // signOff() if not already done
  removeChannels();
}

void GLMoleculeObject_bond::removeChannels()
{
  // at this point both atoms should still be alive, hence we may safely sign off
  // from the AtomObservable itself
  if (bond_enabled) {
    if (bondowner != NULL) {
      bondowner->signOff(this, BondObservable::BondRemoved);
      bondowner->signOff(this, BondObservable::DegreeChanged);
    }
    bond_enabled = false;
  }
  if (leftobservable_enabled) {
    if (leftowner != NULL) {
      leftowner->signOff(this, AtomObservable::PositionChanged);
      leftowner->signOff(this, AtomObservable::ElementChanged);
    }
    leftobservable_enabled = false;
  }
  if (rightobservable_enabled) {
    if (rightowner != NULL) {
      rightowner->signOff(this, AtomObservable::PositionChanged);
      rightowner->signOff(this, AtomObservable::ElementChanged);
    }
    rightobservable_enabled = false;
  }
}

void GLMoleculeObject_bond::removeMe()
{
  // sign off
  emit BondRemoved(leftatomId, rightatomId);
}

void GLMoleculeObject_bond::update(Observable *publisher)
{
  ASSERT(0, "GLMoleculeObject_bond::update() - we are not signed on for any global updates.");
}

void GLMoleculeObject_bond::subjectKilled(Observable *publisher)
{
  // we signOff from all other sources
  removeChannels();
  // then indicate to remove us
  removeMe();
}

void GLMoleculeObject_bond::recieveNotification(Observable *publisher, Notification_ptr notification)
{
#ifdef LOG_OBSERVER
  if (publisher == static_cast<const Observable *>(bondowner)) {
    observerLog().addMessage() << "++ Update of Observer "
        << observerLog().getName(static_cast<Observer*>(this))
        << " received notification from bond for channel "
        << notification->getChannelNo() << ".";
  } else if (publisher == static_cast<const Observable * const>(leftowner)) {
    observerLog().addMessage() << "++ Update of Observer "
        << observerLog().getName(static_cast<Observer*>(this))
        << " received notification from leftatom " << leftatomId << " for channel "
        << notification->getChannelNo() << ".";
  } else if (publisher == static_cast<const Observable * const>(rightowner)) {
    observerLog().addMessage() << "++ Update of Observer "
        << observerLog().getName(static_cast<Observer*>(this))
        << " received notification from rightatom " << rightatomId << " for channel "
        << notification->getChannelNo() << ".";
  }
#endif
  if (publisher == static_cast<const Observable *>(bondowner)){
    switch (notification->getChannelNo()) {
    case BondObservable::BondRemoved:
//      removeMe();
      break;
    case BondObservable::DegreeChanged:
      emit degreeChanged();
      break;
    default:
      ASSERT(0, "GLMoleculeObject_bond::recieveNotification() - unknown signal.");
      break;
    }
  } else {
    // from an atom
    switch (notification->getChannelNo()) {
    case AtomObservable::PositionChanged:
      LOG(2, "INFO: Received notification of PositionChanged.");
      emit positionChanged();
      break;
    case AtomObservable::ElementChanged:
      LOG(2, "INFO: Received notification of ElementChanged.");
      emit elementChanged();
      break;
    default:
      break;
    }
  }
}

Vector GLMoleculeObject_bond::updateLeftPosition() const
{
  const atom * const _atom = getAtomConst(leftatomId);
  return _atom->getPosition();
}

Vector GLMoleculeObject_bond::updateRightPosition() const
{
  const atom * const _atom = getAtomConst(rightatomId);
  return _atom->getPosition();
}

atomicNumber_t GLMoleculeObject_bond::updateLeftElement() const
{
  const atom * const _atom = getAtomConst(leftatomId);
  return _atom->getElementNo();
}

atomicNumber_t GLMoleculeObject_bond::updateRightElement() const
{
  const atom * const _atom = getAtomConst(rightatomId);
  return _atom->getElementNo();
}

int GLMoleculeObject_bond::updateDegree() const
{
  const atom * const _leftatom = const_cast<const World &>(World::getInstance()).
      getAtom(AtomById(leftatomId));
  const atom * const _rightatom = const_cast<const World &>(World::getInstance()).
      getAtom(AtomById(rightatomId));
  if ((_leftatom != NULL) && (_rightatom != NULL)) {
    bond::ptr _bond = _leftatom->getBond(_rightatom);
    return _bond->getDegree();
  } else {
    return 1;
  }
}

void GLMoleculeObject_bond::resetElement()
{
  size_t elementno = rightElement.get();
  QGLMaterial *elementmaterial = getMaterial(elementno);
  setMaterial(elementmaterial);
}

void GLMoleculeObject_bond::resetWidth()
{
  const double factor = 1.0f+.5f*(Degree.get()-1);
  LOG(2, "DEBUG: GLMoleculeObject_bond::resetWidth() - setting bond's width to " << factor << ".");
  setScaleX(factor);
  setScaleY(factor);

  emit changed();
}

void GLMoleculeObject_bond::resetPosition()
{
  Vector Position = leftPosition.get();
  Vector OtherPosition = rightPosition.get();
  const double distance =
      Position.distance(OtherPosition)/2.;
  setScaleZ(distance);

  // calculate position
  Vector Z(unitVec[2]); // cylinder are initially aligned along the Z axis
  Vector zeroVec(0.,0.,0.);
  Vector a,b;
  Vector OtherAxis;
  double alpha;
  a = Position - OtherPosition;
  // construct rotation axis
  b = a;
  b.VectorProduct(Z);
  Line axis(zeroVec, b);
  // calculate rotation angle
  alpha = a.Angle(Z);
  // construct other axis to check right-hand rule
  OtherAxis = b;
  OtherAxis.VectorProduct(Z);
  // assure right-hand rule for the rotation
  if (a.ScalarProduct(OtherAxis) < MYEPSILON)
    alpha = M_PI-alpha;
  // check
  Vector a_rotated = axis.rotateVector(a, alpha);
  LOG(3, "INFO: Created cylinder from "// << Position << " to " << OtherPosition
      << a << " to " << a_rotated << " around " << b << " by " << alpha/M_PI*180. << ", respectively.");

  // set position (cylinder offset is in its barymetric center)
  Vector OneFourth(Position - 0.75 * a);
  setPosition(QVector3D(OneFourth[0], OneFourth[1], OneFourth[2]));
  setRotationVector(QVector3D(b[0], b[1], b[2]));
  setRotationAngle(alpha/M_PI*180.);

  emit changed();
}

atom * const GLMoleculeObject_bond::getAtom(const atomId_t _id)
{
  atom * const _atom = World::getInstance().getAtom(AtomById(_id));
  return _atom;
}

const atom * const GLMoleculeObject_bond::getAtomConst(const atomId_t _id)
{
  const atom * const _atom = const_cast<const World &>(World::getInstance()).
      getAtom(AtomById(_id));
  return _atom;
}
