/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2015 Frederik Heber. All rights reserved.
 *
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * TriangleMatcher.cpp
 *
 *  Created on: Jun 15, 2016
 *      Author: heber
 *
 *
 *  This is a small study in C++11 that compares at least two
 *  files containing the triangle index triples of tesselations
 *  for equivalence
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include <boost/exception/all.hpp>
#include <boost/filesystem/operations.hpp>
#include <boost/filesystem/path.hpp>
#include <boost/tokenizer.hpp>

#include <algorithm>
#include <fstream>
#include <iostream>
#include <iterator>
#include <sstream>
#include <string>
#include <map>

#include "types.hpp"

using namespace detail;

namespace bf = boost::filesystem;

typedef boost::error_info<
		struct tag_IllegalTesselationFileContent_LineNumber,
		int> IllegalTesselationFileContent_LineNumber;

typedef boost::error_info<
		struct tag_IllegalTesselationFileContent_Entries,
		int> IllegalTesselationFileContent_Entries;

typedef boost::error_info<
		struct tag_IllegalTesselationFileContent_Line,
		int> IllegalTesselationFileContent_Line;

//!> Exception for illegal lines encountered in tesselation files
struct IllegalTesselationFileContentException:
		virtual boost::exception, virtual std::exception { };

Triangle::index_t toIndex(const std::string &_element)
{
	Triangle::index_t index;
	std::stringstream tokenstring(_element);
	tokenstring >> index;
	return index;
}

template <class T>
std::ostream& operator<<(
		std::ostream &_ost,
		const std::set<T> &elements)
{
	for( const T elem : elements)
		_ost << elem << std::endl;
	return _ost;
}

/** Parses a tesselation file into a vector of Triangles.
 *
 * \param _filename name of file to parse
 * \return tesselation_t structure filled with triangle index triples
 */
const tesselation_t parseFile(const boost::filesystem::path &_filename)
{
	tesselation_t tesselation;

	std::ifstream infile(_filename.string().c_str());
	boost::char_separator<char> sep(" \t");

	typedef boost::tokenizer<boost::char_separator<char> > tokenizer;
	std::string line;
	int line_number = 1;
	// read up till encountering an empty line
	while (!infile.eof()) {
		std::getline(infile, line);
		if (line.empty())
			break;
	}
	while (!infile.eof()) {
		std::getline(infile, line);
		tokenizer tokens(line, sep);
		Triangle::indices_t indices;
		const int entries = std::distance( tokens.begin(), tokens.end() );
		// check for empty lines
		if (entries == 0)
			continue;
		// entries must be exactly three if any
		if ( entries !=  NUMBERINDICESTRIANGLE )
			throw IllegalTesselationFileContentException()
				<< IllegalTesselationFileContent_LineNumber(line_number)
				<< IllegalTesselationFileContent_Entries(entries);
		std::transform(
				tokens.begin(),
				tokens.end(),
				indices.begin(),
				toIndex);
		tesselation.insert(Triangle(indices));

		++line_number;
	}
	infile.close();

	return tesselation;
}

int main(int argc, char **argv) {

	/// check arguments and give usage
	if (argc <= 2) {
		std::cerr << "Usage: " << argv[0] << " <list of files>\n";
		std::cerr << "At least two tesselation files need to be specified that contain\n";
		std::cerr << "each three indices per line.\n";

		return 255;
	}

	/// read in files and parse into structures
	typedef std::multimap<tesselation_t, std::string> tesselations_filename_t;
	//!> contains all tesselations parsed from files
	tesselations_filename_t tesselations;
	for (int i=1; i<argc;++i) {
		const bf::path filename(argv[i]);
		if (bf::exists(filename)) {
			try {
				tesselation_t tesselation = parseFile(filename);
				tesselations.insert(
						std::make_pair(tesselation, filename.string()) );
			} catch (IllegalTesselationFileContentException &e) {
				std::cerr << "Illegal line at "
						<< *boost::get_error_info<IllegalTesselationFileContent_LineNumber>(e)
						<< " encountered with "
						<< *boost::get_error_info<IllegalTesselationFileContent_Entries>(e)
						<< " entries in "
						<< filename.string()
						<< ", aborting.\n";
				return 5;
			}
		} else {
			std::cerr << "The following file does not exist "
					<< filename.string() << ", aborting.\n";
			return 5;
		}
	}

	/// make comparison for equivalence
	assert( tesselations.size() >= 2 );
	tesselations_filename_t::const_iterator advancer = tesselations.begin();
	tesselations_filename_t::const_iterator iter = advancer++;

	bool AllEqual = true;
	while( advancer != tesselations.end() ) {
		if (advancer->first == iter->first) {
			std::cout << "MATCH: "
					<< advancer->second << " and " << iter->second
					<< std::endl;
		} else {
			std::cout << advancer->first << " and " << iter->first
					<< " don't match.\n";
			AllEqual = false;
		}
		iter = advancer++;
	}

	// exit (convert bool to int)
	return (AllEqual ? 0 : 1);
}
