/*
 * Triangle.hpp
 *
 *  Created on: Jun 15, 2015
 *      Author: heber
 */

#ifndef TRIANGLE_HPP_
#define TRIANGLE_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <array>
#include <cassert>
#include <utility>

#include "types.hpp"

namespace detail {

struct Triangle
{
	//!> typedef for the triangle index type
	typedef size_t index_t;

	//!> typedef for the structure to store triangle indices
	typedef std::array<index_t, 3> indices_t;

	//!> constant for invalid index
	enum { INVALIDINDEX=-1 };

	/** Constructor for struct triangle.
	 *
	 */
	Triangle()
	{
		indices = {{
				(index_t)INVALIDINDEX,
				(index_t)INVALIDINDEX,
				(index_t)INVALIDINDEX
		}};
	}

	/** Constructor for struct triangle.
	 *
	 * \param _indices initial indices to set
	 */
	Triangle(const indices_t &_indices) :
		indices(_indices)
	{
		if (!isSorted())
			sortIndices();
	}

	/** Setter for the indices of the triangle.
	 *
	 * \param _elements indices to be set (need not be sorted)
	 */
	void set(const indices_t &_elements) {
		indices = _elements;
		if (!isSorted())
			sortIndices();
	}

	/** Getter for the indices.
	 *
	 * \return set of sorted indices
	 */
	const indices_t &get() const {
		return indices;
	}

	/** Less comparator for triangle, comparing indices.
	 *
	 * \return true - at least one index in this triangle is smaller
	 */
	bool operator<(const Triangle &_other) const {
		return indices < _other.indices;
	}

	/** Comparator for equivalence of triangle.
	 *
	 * \return true - both index triples are the same
	 */
	bool operator==(const Triangle &_other) const {
		return indices == _other.indices;
	}

	/** Comparator for non-equivalence of triangle.
	 *
	 * \return true - both index triples are not the same
	 */
	bool operator!=(const Triangle &_other) const {
		return !(*this == _other);
	}

private:
	/** Sort the indices.
	 *
	 */
	void sortIndices() {
		std::sort(indices.begin(), indices.end());
//		if (indices[0] > indices[1]) {
//			if (indices[0] > indices[2])
//				std::swap(indices[0], indices[2]);
//			if (indices[0] > indices[1])
//				std::swap(indices[0], indices[1]);
//		} else {
//			// last element in between or up front?
//			if (indices[0] > indices[2]) {
//				std::swap(indices[1], indices[2]);
//				std::swap(indices[0], indices[1]);
//			} else
//				std::swap(indices[1], indices[2]);
//		}
		assert( isSorted() );
	}

private:
	/** Check whether indices are sorted.
	 *
	 * \return true - indices are sorted ascendingly
	 */
	bool isSorted() const
	{
		return (( indices[0] <= indices[1]) && (indices[1] <= indices[2]));
	}

private:
	//!> fixed size array for containing indices
	indices_t indices;
};

}; /* namespace detail */

std::ostream& operator<<(std::ostream &_ost, const detail::Triangle &_t) {
	const detail::Triangle::indices_t &indices = _t.get();
	_ost << "[" << indices[0] << ","  << indices[1] << "," << indices[2] << "]";
	return _ost;
}

#endif /* TRIANGLE_HPP_ */
