/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2011 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * \file controller.cpp
 *
 * This file strongly follows the Serialization example from the boost::asio
 * library (see client.cpp)
 *
 *  Created on: Nov 27, 2011
 *      Author: heber
 */


// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// boost asio needs specific operator new
#include <boost/asio.hpp>

#include "CodePatterns/MemDebug.hpp"

#include <iostream>
#include <vector>
#include <map>

#include "atexit.hpp"
#include "CodePatterns/Info.hpp"
#include "CodePatterns/Log.hpp"
#include "Controller/FragmentController.hpp"
#include "Controller/Commands/CheckResultsOperation.hpp"
#include "Controller/Commands/ReceiveJobsOperation.hpp"
#include "Controller/Commands/SendResultsOperation.hpp"
#include "Controller/Commands/ShutdownOperation.hpp"
#include "FragmentJob.hpp"
#include "FragmentResult.hpp"

enum CommandIndices {
  UnknownCommandIndex = 0,
  SendJobsIndex = 1,
  CheckResultsIndex = 2,
  ReceiveResultsIndex = 3,
  ShutdownIndex = 4
};

void createjobs(std::vector<FragmentJob> &jobs)
{
  FragmentJob testJob(std::string("do something"), 1);
  FragmentJob othertestJob(std::string("do something else"), 2);
  jobs.push_back(testJob);
  jobs.push_back(othertestJob);
}

void addjobs(FragmentController &controller, std::vector<FragmentJob> &jobs)
{
  ReceiveJobsOperation *recjobs = static_cast<ReceiveJobsOperation *>(
      controller.Commands.getByName("receivejobs"));
  recjobs->addJobs(jobs);
}

void sendjobs(FragmentController &controller, const std::string &host, const std::string &service)
{
  ReceiveJobsOperation *recjobs = static_cast<ReceiveJobsOperation *>(
      controller.Commands.getByName("receivejobs"));
  (*recjobs)(host, service);
}

void checkresults(FragmentController &controller, const std::string &host, const std::string &service)
{
  CheckResultsOperation *checkres = static_cast<CheckResultsOperation *>(
      controller.Commands.getByName("checkresults"));
  (*checkres)(host, service);
}

void printdonejobs(FragmentController &controller)
{
  CheckResultsOperation *checkres = static_cast<CheckResultsOperation *>(
      controller.Commands.getByName("checkresults"));
  const size_t doneJobs = checkres->getDoneJobs();
  LOG(1, "INFO: " << doneJobs << " jobs are calculated so far.");
}

void receiveresults(FragmentController &controller, const std::string &host, const std::string &service)
{
  SendResultsOperation *sendres = static_cast<SendResultsOperation *>(
      controller.Commands.getByName("sendresults"));
  (*sendres)(host, service);
}

void printreceivedresults(FragmentController &controller)
{
  SendResultsOperation *sendres = static_cast<SendResultsOperation *>(
      controller.Commands.getByName("sendresults"));
  std::vector<FragmentResult> results = sendres->getResults();
  for (std::vector<FragmentResult>::const_iterator iter = results.begin();
      iter != results.end(); ++iter)
    LOG(1, "RESULT: job #"+toString((*iter).getId())+": "+toString((*iter).result));
}

void shutdown(FragmentController &controller, const std::string &host, const std::string &service)
{
  ShutdownOperation *shutdown = static_cast<ShutdownOperation *>(
      controller.Commands.getByName("shutdown"));
  (*shutdown)(host, service);
}

/** Returns a unique index for every command to allow switching over it.
 *
 * \param &commandmap map with command strings
 * \param &cmd command string
 * \return index from CommandIndices: UnkownCommandIndex - unknown command, else - command index
 */
CommandIndices getCommandIndex(std::map<std::string, CommandIndices> &commandmap, const std::string &cmd)
{
  std::map<std::string, CommandIndices>::const_iterator iter = commandmap.find(cmd);
  if (iter != commandmap.end())
    return iter->second;
  else
    return UnknownCommandIndex;
}


int main(int argc, char* argv[])
{
  // from this moment on, we need to be sure to deeinitialize in the correct order
  // this is handled by the cleanup function
  atexit(cleanUp);

  setVerbosity(3);

  size_t Exitflag = 0;
  std::map<std::string, CommandIndices> CommandsMap;
  CommandsMap.insert( std::make_pair("sendjobs", SendJobsIndex) );
  CommandsMap.insert( std::make_pair("checkresults", CheckResultsIndex) );
  CommandsMap.insert( std::make_pair("receiveresults", ReceiveResultsIndex) );
  CommandsMap.insert( std::make_pair("shutdown", ShutdownIndex) );
  try
  {
    // Check command line arguments.
    if (argc != 4)
    {
      std::cerr << "Usage: " << argv[0] << " <host> <port> <command>" << std::endl;
      return 1;
    }

    boost::asio::io_service io_service;
    FragmentController controller(io_service);

    switch(getCommandIndex(CommandsMap, argv[3])) {
      case SendJobsIndex:
      {
        std::vector<FragmentJob> jobs;
        createjobs(jobs);
        addjobs(controller, jobs);
        sendjobs(controller, argv[1], argv[2]);
        break;
      }
      case CheckResultsIndex:
      {
        checkresults(controller, argv[1], argv[2]);
        break;
      }
      case ReceiveResultsIndex:
      {
        receiveresults(controller, argv[1], argv[2]);
        break;
      }
      case ShutdownIndex:
      {
        shutdown(controller, argv[1], argv[2]);
        break;
      }
      case UnknownCommandIndex:
      default:
        ELOG(0, "Unrecognized command '"+toString(argv[3])+"'.");
        break;
    }

    {
      Info info("io_service");
      io_service.run();
    }

    switch(getCommandIndex(CommandsMap, argv[3])) {
      case SendJobsIndex:
        break;
      case CheckResultsIndex:
      {
        printdonejobs(controller);
        break;
      }
      case ReceiveResultsIndex:
      {
        printreceivedresults(controller);
        break;
      }
      case ShutdownIndex:
        break;
      case UnknownCommandIndex:
      default:
        ELOG(0, "Unrecognized command '"+toString(argv[3])+"'.");
        break;
    }
    Exitflag = controller.getExitflag();
  }
  catch (std::exception& e)
  {
    std::cerr << e.what() << std::endl;
  }

  return Exitflag;
}
