/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/** \file FormatParserStorage.cpp
 *
 *  date: Jun, 22 2010
 *  author: heber
 *
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include <iostream>
#include <fstream>

#include "Parser/FormatParserStorage.hpp"

#include "CodePatterns/Log.hpp"
#include "CodePatterns/Verbose.hpp"

#include "CodePatterns/Assert.hpp"

#include "molecule.hpp"

#include "CodePatterns/Singleton_impl.hpp"

/** Increment operator for the enumeration ParserTypes to allow loops.
 * \param &type value
 * \return value incremented by one
 */
ParserTypes &operator++(ParserTypes &type)
{
  return type = ParserTypes(type+1);
}

/** Constructor of class FormatParserStorage.
 */
FormatParserStorage::FormatParserStorage()
{
  ParserList.resize(ParserTypes_end, NULL);
  ParserStream.resize(ParserTypes_end, NULL);
  ParserPresent.resize(ParserTypes_end, false);

  ParserNames[mpqc] = "mpqc";
  ParserNames[pcp] = "pcp";
  ParserNames[pdb] = "pdb";
  ParserNames[tremolo] = "tremolo";
  ParserNames[xyz] = "xyz";

  for (std::map<ParserTypes, std::string>::const_iterator it = ParserNames.begin(); it != ParserNames.end(); ++it)
    ParserLookupNames.insert(pair<std::string, ParserTypes>(it->second,it->first) );

  ParserSuffixes[mpqc] = "in";
  ParserSuffixes[pcp] = "conf";
  ParserSuffixes[pdb] = "pdb";
  ParserSuffixes[tremolo] = "data";
  ParserSuffixes[xyz] = "xyz";

  for (std::map<ParserTypes, std::string>::const_iterator it = ParserSuffixes.begin(); it != ParserSuffixes.end(); ++it)
    ParserLookupSuffixes.insert(pair<std::string, ParserTypes>(it->second,it->first) );

  ParserAddFunction[mpqc] = &FormatParserStorage::addMpqc;
  ParserAddFunction[pcp] = &FormatParserStorage::addPcp;
  ParserAddFunction[pdb] = &FormatParserStorage::addPdb;
  ParserAddFunction[tremolo] = &FormatParserStorage::addTremolo;
  ParserAddFunction[xyz] = &FormatParserStorage::addXyz;
}

/** Destructor of class FormatParserStorage.
 * Free all stored FormatParsers. 
 * Save on Exit.
 */
FormatParserStorage::~FormatParserStorage()
{
  for (ParserTypes iter = ParserTypes_begin; iter < ParserTypes_end; ++iter)
    if (ParserPresent[iter]) {
      if (ParserStream[iter]->is_open())
        ParserStream[iter]->close();
      delete ParserStream[iter];
      delete ParserList[iter];
    }
}

/** Sets the filename of all current parsers in storage to prefix.suffix.
 * \param &prefix prefix to use.
 */
void FormatParserStorage::SetOutputPrefixForAll(std::string &_prefix)
{
  prefix=_prefix;
};


void FormatParserStorage::SaveAll()
{
  std::string filename;
  for (ParserTypes iter = ParserTypes_begin; iter < ParserTypes_end; ++iter)
    if (ParserPresent[iter]) {
      filename = prefix;
      filename += ".";
      filename += ParserSuffixes[iter];
      ParserStream[iter] = new std::ofstream(filename.c_str());
      ParserList[iter]->setOstream((std::ostream *)ParserStream[iter]);
    }
}


/** Adds an MpqcParser to the storage.
 */
void FormatParserStorage::addMpqc()
{
  if (!ParserPresent[mpqc]) {
    ParserList[mpqc] = dynamic_cast<FormatParser *>(new MpqcParser);
    ParserPresent[mpqc] = true;
  }
  else
    DoeLog(2) && (eLog() << Verbose(2) << "Parser mpqc is already present." << endl
        << "Note that you don't need to add '-o mpqc' if the input file is of type mpqc." << endl);
}


/** Adds an PcpParser to the storage.
 */
void FormatParserStorage::addPcp()
{
  if (!ParserPresent[pcp]) {
    ParserList[pcp] = new PcpParser();
    ParserPresent[pcp] = true;
  } else
    DoeLog(2) && (eLog() << Verbose(2) << "Parser pcp is already present." << endl
        << "Note that you don't need to add '-o pcp' if the input file is of type pcp." << endl);
}


/** Adds an PdbParser to the storage.
 */
void FormatParserStorage::addPdb()
{
  if (!ParserPresent[pdb]) {
    ParserList[pdb] = new PdbParser();
    ParserPresent[pdb] = true;
  } else
    DoeLog(2) && (eLog() << Verbose(2) << "Parser pdb is already present." << endl
        << "Note that you don't need to add '-o pdb' if the input file is of type pdb." << endl);
}


/** Adds an TremoloParser to the storage.
 */
void FormatParserStorage::addTremolo()
{
  if (!ParserPresent[tremolo]) {
    ParserList[tremolo] = new TremoloParser();
    ParserPresent[tremolo] = true;
  } else
    DoeLog(2) && (eLog() << Verbose(2) << "Parser tremolo is already present." << endl
        << "Note that you don't need to add '-o tremolo' if the input file is of type tremolo." << endl);
}


/** Adds an XyzParser to the storage.
 */
void FormatParserStorage::addXyz()
{
  if (!ParserPresent[xyz]) {
    ParserList[xyz] = new XyzParser();
    ParserPresent[xyz] = true;
  } else
    DoeLog(2) && (eLog() << Verbose(2) << "Parser xyz is already present." << endl
        << "Note that you don't need to add '-o xyz' if the input file is of type xyz." << endl);
}

ParserTypes FormatParserStorage::getTypeFromName(std::string type)
{
  if (ParserLookupNames.find(type) == ParserLookupNames.end()) {
    DoeLog(1) && (eLog() << Verbose(1) << "Unknown type " << type << "." << endl);
    return ParserTypes_end;
  } else
    return ParserLookupNames[type];
}

ParserTypes FormatParserStorage::getTypeFromSuffix(std::string type)
{
  if (ParserLookupSuffixes.find(type) == ParserLookupSuffixes.end()) {
    DoeLog(1) && (eLog() << Verbose(1) << "Unknown type " << type << "." << endl);
    return ParserTypes_end;
  } else
    return ParserLookupSuffixes[type];
}

bool FormatParserStorage::add(ParserTypes ptype)
{
  if (ptype != ParserTypes_end) {
    if (ParserAddFunction.find(ptype) != ParserAddFunction.end()) {
      DoLog(0) && (Log() << Verbose(0) << "Adding " << ParserNames[ptype] << " type to output." << endl);
      (getInstance().*(ParserAddFunction[ptype]))(); // we still need an object to work on ...
      return true;
    } else {
      DoeLog(1) && (eLog() << Verbose(1) << "No parser to add for this known type " << ParserNames[ptype] << ", not implemented?" << endl);
      return false;
    }
  } else {
    return false;
  }
}

bool FormatParserStorage::add(std::string type)
{
  return add(getTypeFromName(type));
}


/** Parses an istream depending on its suffix
 * \param &input input stream
 * \param suffix
 * \return true - parsing ok, false - suffix unknown
 */
bool FormatParserStorage::load(std::istream &input, std::string suffix)
{
  if (suffix == ParserSuffixes[mpqc]) {
    getMpqc().load(&input);
  } else if (suffix == ParserSuffixes[pcp]) {
    getPcp().load(&input);
  } else if (suffix == ParserSuffixes[pdb]) {
    getPdb().load(&input);
  } else if (suffix == ParserSuffixes[tremolo]) {
    getTremolo().load(&input);
  } else if (suffix == ParserSuffixes[xyz]) {
    getXyz().load(&input);
  } else {
    DoeLog(1) && (eLog() << Verbose(1) << "Unknown suffix " << suffix << " to for FormatParserStorage::get()." << endl);
    return false;
  }
  return true;
}

/** Stores all selected atoms in an ostream depending on its suffix
 * \param &output output stream
 * \param suffix
 * \return true - storing ok, false - suffix unknown
 */
bool FormatParserStorage::saveSelectedAtoms(std::ostream &output, std::string suffix)
{
  std::vector<atom *> atoms = World::getInstance().getSelectedAtoms();
  return save(output, suffix, atoms);
}

/** Stores all selected atoms in an ostream depending on its suffix
 * We store in the order of the atomic ids, not in the order they appear in the molecules.
 * Hence, we first create a vector from all selected molecules' atoms.
 * \param &output output stream
 * \param suffix
 * \return true - storing ok, false - suffix unknown
 */
bool FormatParserStorage::saveSelectedMolecules(std::ostream &output, std::string suffix)
{
  std::vector<molecule *> molecules = World::getInstance().getSelectedMolecules();
  std::map<size_t, atom *> IdAtoms;
  for (std::vector<molecule *>::const_iterator MolIter = molecules.begin();
      MolIter != molecules.end();
      ++MolIter) {
    for(molecule::atomSet::const_iterator AtomIter = (*MolIter)->begin();
        AtomIter != (*MolIter)->end();
        ++AtomIter) {
      IdAtoms.insert( make_pair((*AtomIter)->getId(), (*AtomIter)) );
    }
  }
  std::vector<atom *> atoms;
  atoms.reserve(IdAtoms.size());
  for (std::map<size_t, atom *>::const_iterator iter = IdAtoms.begin();
      iter != IdAtoms.end();
      ++iter) {
    atoms.push_back(iter->second);
  }
  return save(output, suffix, atoms);
}

/** Stores world in an ostream depending on its suffix
 * \param &output output stream
 * \param suffix
 * \return true - storing ok, false - suffix unknown
 */
bool FormatParserStorage::saveWorld(std::ostream &output, std::string suffix)
{
  std::vector<atom *> atoms = World::getInstance().getAllAtoms();
  return save(output, suffix, atoms);
}

/** Stores a given vector of \a atoms in an ostream depending on its suffix
 * \param &output output stream
 * \param suffix
 * \return true - storing ok, false - suffix unknown
 */
bool FormatParserStorage::save(std::ostream &output, std::string suffix, const std::vector<atom *> &atoms)
{
  if (suffix == ParserSuffixes[mpqc]) {
    getMpqc().save(&output, atoms);
  } else if (suffix == ParserSuffixes[pcp]) {
    getPcp().save(&output, atoms);
  } else if (suffix == ParserSuffixes[pdb]) {
    getPdb().save(&output, atoms);
  } else if (suffix == ParserSuffixes[tremolo]) {
    getTremolo().save(&output, atoms);
  } else if (suffix == ParserSuffixes[xyz]) {
    getXyz().save(&output, atoms);
  } else {
    DoeLog(1) && (eLog() << Verbose(1) << "Unknown suffix " << suffix << " to for FormatParserStorage::put()." << endl);
    return false;
  }
  return true;
}

/** Returns reference to the output MpqcParser, adds if not present.
 * \return reference to the output MpqcParser
 */
MpqcParser &FormatParserStorage::getMpqc()
{
  if (!ParserPresent[mpqc])
    addMpqc();
  return dynamic_cast<MpqcParser &>(*ParserList[mpqc]);
}

/** Returns reference to the output PcpParser, adds if not present.
 * \return reference to the output PcpParser
 */
PcpParser &FormatParserStorage::getPcp()
{
  if (!ParserPresent[pcp])
    addPcp();
  return dynamic_cast<PcpParser &>(*ParserList[pcp]);
}

/** Returns reference to the output PdbParser, adds if not present.
 * \return reference to the output PdbParser
 */
PdbParser &FormatParserStorage::getPdb()
{
  if (!ParserPresent[pdb])
    addPdb();
  return dynamic_cast<PdbParser &>(*ParserList[pdb]);
}

/** Returns reference to the output TremoloParser, adds if not present.
 * \return reference to the output TremoloParser
 */
TremoloParser &FormatParserStorage::getTremolo()
{
  if (!ParserPresent[tremolo])
    addTremolo();
  return dynamic_cast<TremoloParser &>(*ParserList[tremolo]);
}

/** Returns reference to the output XyzParser, adds if not present.
 * \return reference to the output XyzParser
 */
XyzParser &FormatParserStorage::getXyz()
{
  if (!ParserPresent[xyz])
    addXyz();
  return dynamic_cast<XyzParser &>(*ParserList[xyz]);
}

/** Returns reference to the desired output parser as FormatParser, adds if not present.
 * \param _type type of desired parser
 * \return reference to the output FormatParser with desired type
 */
FormatParser &FormatParserStorage::get(ParserTypes _type)
{
  if (!ParserPresent[_type]) {
    add(_type);
  }
  return *ParserList[_type];
}

CONSTRUCT_SINGLETON(FormatParserStorage)
