/*
 * HomologyContainer.hpp
 *
 *  Created on: Sep 22, 2012
 *      Author: heber
 */

#ifndef HOMOLOGYCONTAINER_HPP_
#define HOMOLOGYCONTAINER_HPP_


// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <boost/serialization/export.hpp>
#include <boost/serialization/map.hpp>
#include <boost/serialization/vector.hpp>

#include <iosfwd>
#include <map>
#include <vector>

#include "Fragmentation/Homology/HomologyGraph.hpp"
#include "Fragmentation/SetValues/Fragment.hpp"

class HomologyContainerTest;

/** This class takes all KeySets in a Graph, checks for those that homologues
 * of one another and places them together.
 *
 * This is meant as a storage for key, value pairs, where the key is the KeySet
 * and the value is the energy associated to the fragment this keyset
 * represents.
 * Afterwards this can then be used as training data for a high-dimensional
 * approximation to the Born-Oppenheimer-surface decomposed into lower-
 * dimensional terms in an ANOVA-like fashion.
 *
 */
class HomologyContainer
{
  //!> grant access to output operator
  friend std::ostream& operator<<(std::ostream &out, const HomologyContainer &container);
  //!> grant unit test access
  friend class HomologyContainerTest;
public:
  typedef double energy_t;
  typedef std::pair<Fragment, energy_t> value_t;
  typedef std::multimap< HomologyGraph, value_t> container_t;
  typedef std::pair< container_t::const_iterator, container_t::const_iterator> range_t;
public:
  /** Default Constructor of class HomologyContainer.
   *
   */
  HomologyContainer() {}

  /** Constructor of class HomologyContainer.
   *
   * @param values values with with to initially fill the container
   */
  HomologyContainer(const container_t &values) :
    container(values)
  {}
  /** Destructor of class HomologyContainer.
   *
   */
  ~HomologyContainer() {}

  /** Equality comparator.
   *
   * Sadly, the insertion order of a std::multimap's values is not guaranteed
   * by the standard and boost::serialization does not heed the ordering of
   * the values associated to the same key. Hence, we implement a weaker
   * comparator for this class in order for the unit test to pass as we don't
   * actuallty care about the order of the homologous fragments.
   *
   * @param other instance to compare to
   * @return true - each container contains all elements of the other
   */
  bool operator==(const HomologyContainer &other) const {
    return ((*this >= other) && (other >= *this));
  }
  bool operator!=(const HomologyContainer& other) const {
    return !(*this == other);
  }

  /** Greater equal comparator, i.e. subset comparator
   *
   * @param other container to check if it's subset
   * @return true - \a other is a subset of this
   */
  bool operator>=(const HomologyContainer &other) const;

  /** Inserter for more graphs along with values.
   *
   * @param values graph and values to insert
   */
  void insert(const container_t &values) {
    container.insert(values.begin(), values.end());
  }

  /** Returns iterator range with all contained graphs homologous to the given \a graph.
   *
   * @param graph graph to match
   * @return iterator range with all matches
   */
  range_t getHomologousGraphs(const HomologyGraph &graph) {
    return container.equal_range(graph);
  }

private:
  //!> multimap containing all homologous graph under same key but each with its value
  container_t container;

private:
  friend class boost::serialization::access;
  // serialization
  template <typename Archive>
  void serialize(Archive& ar, const unsigned int version)
  {
    ar & container;
  }
};

/** Output operator for HomologyContainer.
 *
 * \param out output stream
 * \param container container to print
 * \return output stream for concatenation
 */
std::ostream& operator<<(std::ostream &out, const HomologyContainer &container);

// we need to give this class a unique key for serialization
BOOST_CLASS_EXPORT_KEY(HomologyContainer)


#endif /* HOMOLOGYCONTAINER_HPP_ */
