/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * VectorQtQuery.cpp
 *
 *  Created on: Oct 25, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <QtGui/QDoubleSpinBox>
#include <Qt/qboxlayout.h>
#include <Qt/qcombobox.h>
#include <Qt/qlabel.h>
#include <Qt/qstackedwidget.h>

//#include "CodePatterns/MemDebug.hpp"

#include "UIElements/Qt4/Query/QtQuery.hpp"

#include "CodePatterns/toString.hpp"

#include "Geometry/GeometryRegistry.hpp"
#include "Parameters/Specifics/Value_vector.hpp"

QtDialog::VectorQtQuery::VectorQtQuery(Parameter<Vector> &_param, const std::string &_title, const std::string &_description, QBoxLayout *_parent,Dialog *_dialog) :
    QtQuery<Vector>(_param, _title, _description),
    parent(_parent),
    dialog(_dialog)
{
  Vector temporary(0, 0, 0);
  temp = "0, 0, 0";
  if (param.isSet()) {
    temporary = param.get();
    temp = param.getAsString();
  }
  mainLayout= new QHBoxLayout();
  titleLabel = new QLabel(QString(getTitle().c_str()));
  titleLabel->setToolTip(QString(getDescription().c_str()));
  mainLayout->addWidget(titleLabel);
  subLayout = new QHBoxLayout();
  mainLayout->addLayout(subLayout);
//  QComboBox* inputBox = new QComboBox();

  QWidget *firstPageWidget = new QWidget;
  QWidget *secondPageWidget = new QWidget;

  QStackedWidget *stackedWidget = new QStackedWidget;
  stackedWidget->addWidget(firstPageWidget);
  stackedWidget->addWidget(secondPageWidget);

  QComboBox *pageComboBox = new QComboBox;
  pageComboBox->addItem(tr("x,y,z"));
  pageComboBox->addItem(tr("vector name"));
  connect(pageComboBox, SIGNAL(activated(int)),
      stackedWidget, SLOT(setCurrentIndex(int)));
  connect(pageComboBox, SIGNAL(activated(int)),
      this, SLOT(pageChanged(int)));
  subLayout->addWidget(pageComboBox);
  subLayout->addWidget(stackedWidget);

   // first widget with coordinates
  coordLayout = new QHBoxLayout();
  coordInputX = new QDoubleSpinBox();
  coordInputX->setRange(-std::numeric_limits<double>::max(),std::numeric_limits<double>::max());
  coordInputX->setValue(temporary[0]);
//  coordInputX->setRange(0,M.at(i,i));
  coordInputX->setDecimals(3);
  coordLayout->addWidget(coordInputX);
  coordInputY = new QDoubleSpinBox();
  coordInputY->setRange(-std::numeric_limits<double>::max(),std::numeric_limits<double>::max());
  coordInputY->setValue(temporary[1]);
//  coordInputY->setRange(0,M.at(i,i));
  coordInputY->setDecimals(3);
  coordLayout->addWidget(coordInputY);
  coordInputZ = new QDoubleSpinBox();
  coordInputZ->setRange(-std::numeric_limits<double>::max(),std::numeric_limits<double>::max());
  coordInputZ->setValue(temporary[2]);
//  coordInputZ->setRange(0,M.at(i,i));
  coordInputZ->setDecimals(3);
  coordLayout->addWidget(coordInputZ);
  connect(coordInputX,SIGNAL(valueChanged(double)),this,SLOT(onUpdateX(double)));
  connect(coordInputY,SIGNAL(valueChanged(double)),this,SLOT(onUpdateY(double)));
  connect(coordInputZ,SIGNAL(valueChanged(double)),this,SLOT(onUpdateZ(double)));
  firstPageWidget->setLayout(coordLayout);

  // second widget with string field
  nameLayout = new QHBoxLayout();
  nameComboBox = new QComboBox;
  GeometryRegistry &reg = GeometryRegistry::getInstance();
//  nameComboBox->setEditable(true);
  GeometryRegistry::const_iterator iter;
  for (iter = reg.getBeginIter(); iter != reg.getEndIter(); iter ++){
    GeometryObject *v = iter->second;
    nameComboBox->addItem(tr(v->getName().c_str()));
    nameComboBox->setItemData(nameComboBox->count()-1, tr(toString<Vector>(v->getVector()).c_str()), Qt::ToolTipRole);
  }
  connect(nameComboBox, SIGNAL(activated(int)),
      this, SLOT(onUpdateName(int)));
  nameLayout->addWidget(nameComboBox);
  secondPageWidget->setLayout(nameLayout);

  parent->addLayout(mainLayout);
}

QtDialog::VectorQtQuery::~VectorQtQuery()
{}

static void updateVectorString(std::string &_temp, const double newDouble, int component)
{
  //!> Internal converter from string to internal type
  Vector vec = Value<Vector>::parseAsVector(_temp);
  vec[component] = newDouble;
  _temp = Value<Vector>::setFromVector(vec);
}

void QtDialog::VectorQtQuery::pageChanged(int pagenr) {
  if (pagenr == 1) {
    // change from x,y,z input
    onUpdateName(nameComboBox->currentIndex());
    dialog->update();
  } else if (pagenr == 0) {
    // change from name input
    if (GeometryRegistry::getInstance().isPresentByName(temp)) {
      const GeometryObject * const v = GeometryRegistry::getInstance().getByName(temp);
      coordInputX->setValue(v->getVector()[0]);
      coordInputY->setValue(v->getVector()[1]);
      coordInputZ->setValue(v->getVector()[2]);
    } else {
      coordInputX->setValue(0.);
      coordInputY->setValue(0.);
      coordInputZ->setValue(0.);
    }
    dialog->update();
  } else {
    ASSERT(0, "VectorQtQuery::pageChanged() - unknown page for pageComboBox.");
  }
}

void QtDialog::VectorQtQuery::onUpdateName(int index) {
  const QString itemtext = nameComboBox->itemText(index);
  temp = itemtext.toStdString();
  dialog->update();
}

void QtDialog::VectorQtQuery::onUpdateX(double newDouble) {
  updateVectorString(temp, newDouble, 0);
  dialog->update();
}

void QtDialog::VectorQtQuery::onUpdateY(double newDouble) {
  updateVectorString(temp, newDouble, 1);
  dialog->update();
}

void QtDialog::VectorQtQuery::onUpdateZ(double newDouble) {
  updateVectorString(temp, newDouble, 2);
  dialog->update();
}


