/*
 * TextDialog.cpp
 *
 *  Created on: Jan 5, 2010
 *      Author: crueger
 */

#include "Helpers/MemDebug.hpp"

#include <iostream>

#include <Descriptors/AtomDescriptor.hpp>
#include <Descriptors/AtomIdDescriptor.hpp>
#include <Descriptors/MoleculeDescriptor.hpp>
#include <Descriptors/MoleculeIdDescriptor.hpp>
#include "TextUI/TextDialog.hpp"

#include "World.hpp"
#include "periodentafel.hpp"
#include "log.hpp"
#include "verbose.hpp"

#include "atom.hpp"
#include "element.hpp"
#include "molecule.hpp"
#include "vector.hpp"
#include "Matrix.hpp"
#include "Box.hpp"

using namespace std;


TextDialog::TextDialog()
{
}

TextDialog::~TextDialog()
{
}


void TextDialog::queryEmpty(const char* title, string description){
  registerQuery(new EmptyTextQuery(title,description));
}

void TextDialog::queryBoolean(const char* title, bool* target, string description){
  registerQuery(new BooleanTextQuery(title,target,description));
}

void TextDialog::queryInt(const char* title, int* target, string description){
  registerQuery(new IntTextQuery(title,target,description));
}

void TextDialog::queryDouble(const char* title, double* target, string description){
  registerQuery(new DoubleTextQuery(title,target,description));
}

void TextDialog::queryString(const char* title, string* target, string description){
  registerQuery(new StringTextQuery(title,target,description));
}

void TextDialog::queryStrings(const char* title, vector<string>* target, string description){
  registerQuery(new StringsTextQuery(title,target,description));
}

void TextDialog::queryAtom(const char* title, atom **target, string description) {
  registerQuery(new AtomTextQuery(title,target,description));
}

void TextDialog::queryMolecule(const char* title, molecule **target, string description) {
  registerQuery(new MoleculeTextQuery(title,target,description));
}

void TextDialog::queryVector(const char* title, Vector *target, bool check, string description) {
  registerQuery(new VectorTextQuery(title,target,check,description));
}

void TextDialog::queryBox(const char* title,Box* cellSize, string description) {
  registerQuery(new BoxTextQuery(title,cellSize,description));
}

void TextDialog::queryElement(const char* title, std::vector<element *> *target, string description){
  registerQuery(new ElementTextQuery(title,target,description));
}

/************************** Query Infrastructure ************************/

TextDialog::EmptyTextQuery::EmptyTextQuery(string title, std::string _description) :
    Dialog::EmptyQuery(title,_description)
{}

TextDialog::EmptyTextQuery::~EmptyTextQuery() {}

bool TextDialog::EmptyTextQuery::handle() {
  cout << "Message of " << getTitle() << ":\n" << getDescription() << "\n";
  return true;
}

TextDialog::IntTextQuery::IntTextQuery(string title, int * _target, std::string _description) :
    Dialog::IntQuery(title,_target,_description)
{}

TextDialog::IntTextQuery::~IntTextQuery() {}

bool TextDialog::IntTextQuery::handle() {
  bool badInput = false;
  do{
    badInput = false;
    Log() << Verbose(0) << getTitle();
    cin >> tmp;
    if(cin.fail()){
      badInput=true;
      cin.clear();
      cin.ignore(std::numeric_limits<streamsize>::max(),'\n');
      Log() << Verbose(0) << "Input was not a number!" << endl;
    }
  } while(badInput);
  // clear the input buffer of anything still in the line
  cin.ignore(std::numeric_limits<streamsize>::max(),'\n');
  return true;
}

TextDialog::BooleanTextQuery::BooleanTextQuery(string title, bool * _target, std::string _description) :
    Dialog::BooleanQuery(title,_target,_description)
{}

TextDialog::BooleanTextQuery::~BooleanTextQuery() {}

bool TextDialog::BooleanTextQuery::handle() {
  bool badInput = false;
  char input = ' ';
  do{
    badInput = false;
    Log() << Verbose(0) << getTitle();
    cin >> input;
    if ((input == 'y' ) || (input == 'Y')) {
      tmp = true;
    } else if ((input == 'n' ) || (input == 'N')) {
      tmp = false;
    } else {
      badInput=true;
      cin.clear();
      cin.ignore(std::numeric_limits<streamsize>::max(),'\n');
      Log() << Verbose(0) << "Input was not of [yYnN]!" << endl;
    }
  } while(badInput);
  // clear the input buffer of anything still in the line
  cin.ignore(std::numeric_limits<streamsize>::max(),'\n');
  return true;
}

TextDialog::StringTextQuery::StringTextQuery(string title,string *_target, std::string _description) :
    Dialog::StringQuery(title,_target,_description)
{}

TextDialog::StringTextQuery::~StringTextQuery() {}

bool TextDialog::StringTextQuery::handle() {
  Log() << Verbose(0) << getTitle();
  getline(cin,tmp);
  return true;
}

TextDialog::StringsTextQuery::StringsTextQuery(string title,vector<string> *_target, std::string _description) :
    Dialog::StringsQuery(title,_target,_description)
{}

TextDialog::StringsTextQuery::~StringsTextQuery() {}

bool TextDialog::StringsTextQuery::handle() {
  Log() << Verbose(0) << getTitle();
  getline(cin,temp);
  // dissect by ","
  string::iterator olditer = temp.begin();
  for(string::iterator iter = temp.begin(); iter != temp.end(); ++iter) {
    if (*iter == ' ') {
      tmp.push_back(string(iter, olditer));
      olditer = iter;
    }
  }
  if (olditer != temp.begin())  // insert last part also
    tmp.push_back(string(olditer, temp.end()));

  return true;
}

TextDialog::DoubleTextQuery::DoubleTextQuery(string title,double *_target, std::string _description) :
    Dialog::DoubleQuery(title,_target,_description)
{}

TextDialog::DoubleTextQuery::~DoubleTextQuery() {}

bool TextDialog::DoubleTextQuery::handle() {
  bool badInput = false;
  do{
    badInput = false;
    Log() << Verbose(0) << getTitle();
    cin >> tmp;
    if(cin.fail()){
      badInput = true;
      cin.clear();
      cin.ignore(std::numeric_limits<streamsize>::max(),'\n');
      Log() << Verbose(0) << "Input was not a number!" << endl;
    }
  }while(badInput);
  cin.ignore(std::numeric_limits<streamsize>::max(),'\n');
  return true;
}

TextDialog::AtomTextQuery::AtomTextQuery(string title, atom **_target, std::string _description) :
    Dialog::AtomQuery(title,_target,_description)
{}

TextDialog::AtomTextQuery::~AtomTextQuery() {}

bool TextDialog::AtomTextQuery::handle() {
  int idxOfAtom=0;
  bool badInput = false;
  do{
    badInput = false;
    Log() << Verbose(0) << getTitle();
    cin >> idxOfAtom;
    if(cin.fail()){
      badInput = true;
      cin.clear();
      cin.ignore(std::numeric_limits<streamsize>::max(),'\n');
      Log() << Verbose(0) << "Input was not a number!" << endl;
      continue;
    }

    tmp = World::getInstance().getAtom(AtomById(idxOfAtom));
    if(!tmp && idxOfAtom!=-1){
      Log() << Verbose(0) << "Invalid Atom Index" << endl;
      badInput = true;
    }

  } while(badInput);
  cin.ignore(std::numeric_limits<streamsize>::max(),'\n');
  return (idxOfAtom!=-1);
}

TextDialog::MoleculeTextQuery::MoleculeTextQuery(string title, molecule **_target, std::string _description) :
    Dialog::MoleculeQuery(title,_target,_description)
{}

TextDialog::MoleculeTextQuery::~MoleculeTextQuery() {}

bool TextDialog::MoleculeTextQuery::handle() {
  int idxOfMol=0;
  bool badInput = false;
  do{
    badInput = false;
    Log() << Verbose(0) << getTitle();
    cin >> idxOfMol;
    if(cin.fail()){
      badInput = true;
      cin.clear();
      cin.ignore(std::numeric_limits<streamsize>::max(),'\n');
      Log() << Verbose(0) << "Input was not a number!" << endl;
      continue;
    }

    tmp = World::getInstance().getMolecule(MoleculeById(idxOfMol));
    if(!tmp && idxOfMol!=-1){
      Log() << Verbose(0) << "Invalid Molecule Index" << endl;
      badInput = true;
    }

  } while(badInput);
  cin.ignore(std::numeric_limits<streamsize>::max(),'\n');
  return (idxOfMol!=-1);
}

TextDialog::VectorTextQuery::VectorTextQuery(std::string title, Vector *_target, bool _check, std::string _description) :
    Dialog::VectorQuery(title,_target,_check,_description)
{}

TextDialog::VectorTextQuery::~VectorTextQuery()
{}

bool TextDialog::VectorTextQuery::handle() {
  Log() << Verbose(0) << getTitle();

  const Matrix &M = World::getInstance().getDomain().getM();
  char coords[3] = {'x','y','z'};
  for (int i=0;i<3;i++) {
    do {
      Log() << Verbose(0) << coords[i] << "[0.." << M.at(i,i) << "]: ";
      cin >> (*tmp)[i];
    } while ((((*tmp)[i] < 0) || ((*tmp)[i] >= M.at(i,i))) && (check));
  }
  return true;
}

TextDialog::BoxTextQuery::BoxTextQuery(std::string title, Box* _cellSize, std::string _description) :
    Dialog::BoxQuery(title,_cellSize,_description)
{}

TextDialog::BoxTextQuery::~BoxTextQuery()
{}

bool TextDialog::BoxTextQuery::handle() {
  Log() << Verbose(0) << getTitle();

  std::string coords[6] = {"xx","xy","xz", "yy", "yz", "zz"};
  for (int i=0;i<6;i++) {
    Log() << Verbose(0) << coords[i] << ": ";
    cin >> tmp[i];
  }
  return true;
}

TextDialog::ElementTextQuery::ElementTextQuery(std::string title, std::vector<element *> *_target, std::string _description) :
    Dialog::ElementQuery(title,_target,_description)
{}

TextDialog::ElementTextQuery::~ElementTextQuery()
{}

bool TextDialog::ElementTextQuery::handle() {
  bool badInput=false;
  bool aborted = false;
  element * tmp = NULL;
  do{
    badInput = false;
    Log() << Verbose(0) << getTitle();

    // try to read as Atomic number
    int Z;
    cin >> Z;
    if(!cin.fail()){
      if(Z==-1){
        aborted = true;
      }
      else{
        tmp = World::getInstance().getPeriode()->FindElement(Z);
        if(!tmp){
          Log() << Verbose(0) << "No element with this atomic number!" << endl;
          badInput = true;
        } else {
          elements.push_back(tmp);
        }
      }
      continue;
    }
    else{
      cin.clear();
    }

    // Try to read as shorthand
    // the last buffer content was not removed, so we read the
    // same thing again, this time as a string
    string shorthand;
    cin >> shorthand;
    if(!cin.fail()){
      if(shorthand.empty()){
        aborted = true;
      }
      else{
        tmp = World::getInstance().getPeriode()->FindElement(shorthand.c_str());
        if(!tmp){
          Log() << Verbose(0) << "No element with this shorthand!" << endl;
          badInput = true;
        } else {
          elements.push_back(tmp);
        }
      }
    }
    else{
      Log() << Verbose(0) << "Could not read input. Try Again." << endl;
      cin.clear();
      cin.ignore(std::numeric_limits<streamsize>::max(),'\n');
      badInput = true;
    }

  }while(badInput);
  cin.ignore(std::numeric_limits<streamsize>::max(),'\n');
  return !aborted;
}
