/*
 * QTDialog.cpp
 *
 *  Created on: Jan 18, 2010
 *      Author: crueger
 */

#include "UIElements/QT4/QTDialog.hpp"

#include <string>
#include <sstream>
#include <limits>

#include <Qt/qboxlayout.h>
#include <Qt/qlabel.h>
#include <Qt/qspinbox.h>
#include <QtGui/QDoubleSpinBox>
#include <Qt/qlineedit.h>
#include <Qt/qdialogbuttonbox.h>
#include <Qt/qpushbutton.h>
#include <Qt/qcombobox.h>

#include "Helpers/MemDebug.hpp"

#include "World.hpp"
#include "periodentafel.hpp"
#include "atom.hpp"
#include "element.hpp"
#include "molecule.hpp"
#include "Descriptors/MoleculeIdDescriptor.hpp"
#include "Matrix.hpp"
#include "Box.hpp"


using namespace std;

QTDialog::QTDialog() :
    QDialog(0)
{
  // creating and filling of the Dialog window
  mainLayout = new QVBoxLayout();
  inputLayout = new QVBoxLayout();
  buttonLayout = new QVBoxLayout();
  setLayout(mainLayout);
  mainLayout->addLayout(inputLayout);
  mainLayout->addLayout(buttonLayout);
  buttons = new QDialogButtonBox(QDialogButtonBox::Ok| QDialogButtonBox::Cancel);
  buttonLayout->addWidget(buttons);

  // Disable the ok button until something was entered
  buttons->button(QDialogButtonBox::Ok)->setEnabled(false);

  // connect the buttons to their appropriate slots
  connect(buttons, SIGNAL(accepted()), this, SLOT(accept()));
  connect(buttons, SIGNAL(rejected()), this, SLOT(reject()));
}

QTDialog::~QTDialog()
{
}

bool QTDialog::display(){
  // Button state might have changed by some update that
  // was done during query construction. To make sure
  // the state is correct, we just call update one more time.
  update();
  if(exec()) {
    setAll();
    return true;
  }
  else {
    return false;
  }
}

void QTDialog::update(){
  buttons->button(QDialogButtonBox::Ok)->setEnabled(checkAll());
}

/************************** Query Infrastructure ************************/

void QTDialog::queryEmpty(char const*, string){
  // TODO
  ASSERT(false, "Not implemented yet");
}

void QTDialog::queryBoolean(char const*, bool*,string){
  // TODO
  ASSERT(false, "Not implemented yet");
}

void QTDialog::queryAtom(char const*, atom**, string){
  // TODO
  ASSERT(false, "Not implemented yet");
}

void QTDialog::queryBox(char const*, Box*, string){
  // TODO
  ASSERT(false, "Not implemented yet");
}


void QTDialog::queryInt(const char *title, int *target,string)
{
  registerQuery(new IntQTQuery(title,target,inputLayout,this));
}

void QTDialog::queryDouble(const char* title, double* target,string){
  registerQuery(new DoubleQTQuery(title,target,inputLayout,this));
}

void QTDialog::queryString(const char* title, std::string *target,string)
{
  registerQuery(new StringQTQuery(title,target,inputLayout,this));
}

void QTDialog::queryStrings(const char* title, std::vector<std::string> *target,string)
{
  registerQuery(new StringsQTQuery(title,target,inputLayout,this));
}

void QTDialog::queryMolecule(const char *title,molecule **target,string)
{
  registerQuery(new MoleculeQTQuery(title,target,inputLayout,this));
}

void QTDialog::queryVector(const char* title, Vector *target, bool check,string) {
  registerQuery(new VectorQTQuery(title,target,check,inputLayout,this));
}

void QTDialog::queryElement(const char* title, std::vector<element *> *target,string){
  registerQuery(new ElementQTQuery(title,target,inputLayout,this));
}

/************************** Query Objects *******************************/

QTDialog::IntQTQuery::IntQTQuery(string _title,int *_target,QBoxLayout *_parent,QTDialog *_dialog) :
    Dialog::IntQuery(_title,_target),
    parent(_parent)
{
  thisLayout = new QHBoxLayout();
  titleLabel = new QLabel(QString(getTitle().c_str()));
  inputBox = new QSpinBox();
  inputBox->setValue(0);
  parent->addLayout(thisLayout);
  thisLayout->addWidget(titleLabel);
  thisLayout->addWidget(inputBox);

  pipe = new IntQTQueryPipe(&tmp,_dialog);
  pipe->update(inputBox->value());
  connect(inputBox,SIGNAL(valueChanged(int)),pipe,SLOT(update(int)));
}

QTDialog::IntQTQuery::~IntQTQuery()
{
  delete pipe;
}

// Handling is easy since the GUI makes it impossible to enter invalid values
bool QTDialog::IntQTQuery::handle()
{
  return true;
}

QTDialog::DoubleQTQuery::DoubleQTQuery(string title,double *_target,QBoxLayout *_parent,QTDialog *_dialog) :
    Dialog::DoubleQuery(title,_target),
    parent(_parent)
{
  thisLayout = new QHBoxLayout();
  titleLabel = new QLabel(QString(getTitle().c_str()));
  inputBox = new QDoubleSpinBox();
  inputBox->setValue(0);
  inputBox->setRange(-numeric_limits<double>::max(),numeric_limits<double>::max());
  inputBox->setDecimals(3);
  parent->addLayout(thisLayout);
  thisLayout->addWidget(titleLabel);
  thisLayout->addWidget(inputBox);

  pipe = new DoubleQTQueryPipe(&tmp,_dialog);
  pipe->update(inputBox->value());
  connect(inputBox,SIGNAL(valueChanged(double)),pipe,SLOT(update(double)));
}

QTDialog::DoubleQTQuery::~DoubleQTQuery()
{
  delete pipe;
}

bool QTDialog::DoubleQTQuery::handle() {
  return true;
}


QTDialog::StringQTQuery::StringQTQuery(string _title,string *_target,QBoxLayout *_parent,QTDialog *_dialog) :
    Dialog::StringQuery(_title,_target),
    parent(_parent)
{
  thisLayout = new QHBoxLayout();
  titleLabel = new QLabel(QString(getTitle().c_str()));
  inputBox = new QLineEdit();
  parent->addLayout(thisLayout);
  thisLayout->addWidget(titleLabel);
  thisLayout->addWidget(inputBox);

  pipe = new StringQTQueryPipe(&tmp,_dialog);
  pipe->update(inputBox->text());
  connect(inputBox,SIGNAL(textChanged(const QString&)),pipe,SLOT(update(const QString&)));
}

QTDialog::StringQTQuery::~StringQTQuery()
{
  delete pipe;
}

// All values besides the empty string are valid
bool QTDialog::StringQTQuery::handle()
{
  return tmp!="";
}

QTDialog::StringsQTQuery::StringsQTQuery(string _title,vector<string> *_target,QBoxLayout *_parent,QTDialog *_dialog) :
    Dialog::StringsQuery(_title,_target),
    parent(_parent)
{
  thisLayout = new QHBoxLayout();
  titleLabel = new QLabel(QString(getTitle().c_str()));
  inputBox = new QLineEdit();
  parent->addLayout(thisLayout);
  thisLayout->addWidget(titleLabel);
  thisLayout->addWidget(inputBox);

  pipe = new StringQTQueryPipe(&temp,_dialog);
  pipe->update(inputBox->text());
  connect(inputBox,SIGNAL(textChanged(const QString&)),pipe,SLOT(update(const QString&)));
}

QTDialog::StringsQTQuery::~StringsQTQuery()
{
  delete pipe;
}

// All values besides the empty string are valid
bool QTDialog::StringsQTQuery::handle()
{
  // dissect by ","
  string::iterator olditer = temp.begin();
  for(string::iterator iter = temp.begin(); iter != temp.end(); ++iter) {
    if (*iter == ' ') {
      tmp.push_back(string(iter, olditer));
      olditer = iter;
    }
  }
  if (olditer != temp.begin())  // insert last part also
    tmp.push_back(string(olditer, temp.end()));

  return temp!="";
}

QTDialog::MoleculeQTQuery::MoleculeQTQuery(string _title, molecule **_target, QBoxLayout *_parent,QTDialog *_dialog) :
    Dialog::MoleculeQuery(_title,_target),
    parent(_parent)
{
  thisLayout = new QHBoxLayout();
  titleLabel = new QLabel(QString(getTitle().c_str()));
  inputBox = new QComboBox();
  // add all molecules to the combo box
  vector<molecule*> molecules = World::getInstance().getAllMolecules();
  for(vector<molecule*>::iterator iter  = molecules.begin();
      iter != molecules.end();
      ++iter) {
    stringstream sstr;
    sstr << (*iter)->IndexNr << "\t" << (*iter)->getName();
    inputBox->addItem(QString(sstr.str().c_str()),QVariant((*iter)->IndexNr));
  }
  parent->addLayout(thisLayout);
  thisLayout->addWidget(titleLabel);
  thisLayout->addWidget(inputBox);

  pipe = new MoleculeQTQueryPipe(&tmp,_dialog,inputBox);
  pipe->update(inputBox->currentIndex());
  connect(inputBox,SIGNAL(currentIndexChanged(int)),pipe,SLOT(update(int)));
}

QTDialog::MoleculeQTQuery::~MoleculeQTQuery()
{
  delete pipe;
}

// Handling is easy, since the GUI makes it impossible to select invalid values
bool QTDialog::MoleculeQTQuery::handle()
{
  return true;
}

QTDialog::VectorQTQuery::VectorQTQuery(std::string title, Vector *_target, bool _check,QBoxLayout *_parent,QTDialog *_dialog) :
    Dialog::VectorQuery(title,_target,_check),
    parent(_parent)
{
  const Matrix& M = World::getInstance().getDomain().getM();
  const char *coords[3] = {"x:","y:","z:"};
  mainLayout= new QHBoxLayout();
  titleLabel = new QLabel(QString(getTitle().c_str()));
  mainLayout->addWidget(titleLabel);
  subLayout = new QVBoxLayout();
  mainLayout->addLayout(subLayout);
  for(int i=0; i<3; i++) {
    coordLayout[i] = new QHBoxLayout();
    subLayout->addLayout(coordLayout[i]);
    coordLabel[i] = new QLabel(QString(coords[i]));
    coordLayout[i]->addWidget(coordLabel[i]);
    coordInput[i] = new QDoubleSpinBox();
    coordInput[i]->setRange(0,M.at(i,i));
    coordInput[i]->setDecimals(3);
    coordLayout[i]->addWidget(coordInput[i]);
    pipe[i] = new DoubleQTQueryPipe(&((*tmp)[i]),_dialog);
    pipe[i]->update(coordInput[i]->value());
    connect(coordInput[i],SIGNAL(valueChanged(double)),pipe[i],SLOT(update(double)));

  }
  parent->addLayout(mainLayout);
}

QTDialog::VectorQTQuery::~VectorQTQuery()
{}

bool QTDialog::VectorQTQuery::handle() {
  return true;
}


QTDialog::ElementQTQuery::ElementQTQuery(std::string _title, vector<element *> *_target, QBoxLayout *_parent, QTDialog *_dialog) :
    Dialog::ElementQuery(_title,_target),
    parent(_parent)
{
  periodentafel *periode = World::getInstance().getPeriode();
  thisLayout = new QHBoxLayout();
  titleLabel = new QLabel(QString(getTitle().c_str()));
  inputBox = new QComboBox();
  for(periodentafel::const_iterator iter = periode->begin();
      iter!=periode->end();
      ++iter)
  {
    stringstream sstr;
    sstr << (*iter).first << "\t" << (*iter).second->name;
    inputBox->addItem(QString(sstr.str().c_str()),QVariant((*iter).first));
  }
  parent->addLayout(thisLayout);
  thisLayout->addWidget(titleLabel);
  thisLayout->addWidget(inputBox);

  pipe = new ElementQTQueryPipe(&elements,_dialog,inputBox);
  pipe->update(inputBox->currentIndex());
  connect(inputBox,SIGNAL(currentIndexChanged(int)),pipe,SLOT(update(int)));
}

QTDialog::ElementQTQuery::~ElementQTQuery()
{
  delete pipe;
}

bool QTDialog::ElementQTQuery::handle(){
  return true;
}

/*************************** Plumbing *******************************/

StringQTQueryPipe::StringQTQueryPipe(string *_content, QTDialog *_dialog) :
  content(_content),
  dialog(_dialog)
{}

StringQTQueryPipe::~StringQTQueryPipe()
{}

void StringQTQueryPipe::update(const QString& newText) {
  content->assign(newText.toStdString());
  dialog->update();
}

IntQTQueryPipe::IntQTQueryPipe(int *_content, QTDialog *_dialog) :
  content(_content),
  dialog(_dialog)
{}

IntQTQueryPipe::~IntQTQueryPipe()
{}

void IntQTQueryPipe::update(int newInt) {
  (*content) = newInt;
  dialog->update();
}

DoubleQTQueryPipe::DoubleQTQueryPipe(double *_content, QTDialog *_dialog) :
  content(_content),
  dialog(_dialog)
{}

DoubleQTQueryPipe::~DoubleQTQueryPipe()
{}

void DoubleQTQueryPipe::update(double newDbl) {
  (*content) = newDbl;
  dialog->update();
}

MoleculeQTQueryPipe::MoleculeQTQueryPipe(molecule **_content, QTDialog *_dialog, QComboBox *_theBox) :
  content(_content),
  dialog(_dialog),
  theBox(_theBox)
{}

MoleculeQTQueryPipe::~MoleculeQTQueryPipe()
{}

void MoleculeQTQueryPipe::update(int newIndex) {
  QVariant data = theBox->itemData(newIndex);
  int idx = data.toInt();
  (*content) = World::getInstance().getMolecule(MoleculeById(idx));
  dialog->update();
}

ElementQTQueryPipe::ElementQTQueryPipe(std::vector<element *> *_content, QTDialog *_dialog, QComboBox *_theBox) :
  content(_content),
  dialog(_dialog),
  theBox(_theBox)
{
  content->resize(1);
}

ElementQTQueryPipe::~ElementQTQueryPipe()
{}

void ElementQTQueryPipe::update(int newIndex) {
  QVariant data = theBox->itemData(newIndex);
  int idx = data.toInt();
  (*content)[0] = World::getInstance().getPeriode()->FindElement(idx);
  dialog->update();
}

