/*
 * Box.hpp
 *
 *  Created on: Jun 30, 2010
 *      Author: crueger
 */

#ifndef BOX_HPP_
#define BOX_HPP_

class Matrix;
class Vector;

#include <list>
#include <vector>
#include "VectorSet.hpp"

/**
 * A simple class that can be used to store periodic Boxes
 * in the form of a parallelepiped.
 *
 * Stores a matrix that can be used to translate from periodic
 * [0,1) space as well as the inverse that is needed to
 * translate back to that space.
 */
class Box
{
public:
  typedef enum{
    Wrap,
    Bounce,
    Ignore
  } BoundaryCondition_t;

  typedef vector<BoundaryCondition_t> Conditions_t;

  Box();
  Box(const Box&);
  virtual ~Box();

  /**
   * Get the matrix describing the form of the parallelepiped
   */
  const Matrix &getM() const;

  /**
   * Get the inverse of the matrix M (see above).
   */
  const Matrix &getMinv() const;

  /**
   * Set the form of the parallelepiped.
   */
  void setM(Matrix);

  Box &operator=(const Box&);
  Box &operator=(const Matrix&);

  /**
   * Translate a point from [0,1) to the boxed space.
   */
  Vector translateIn(const Vector &point) const;

  /**
   * Translate a point from the boxed space to the [0,1) space.
   */
  Vector translateOut(const Vector &point) const;

  /**
   * Wrap a point so that it will lie within the space defined by the box.
   */
  Vector WrapPeriodically(const Vector &point) const;

  /**
   * Checks whether a given vector is inside the box.
   */
  bool isInside(const Vector &point) const;

  /**
   * Produce corresponding points in several adjacent boxes.
   *
   * n specifies the number of times the point is expanded.
   * Carefull, needs O(n^3) time and produces as many vectors.
   */
  VECTORSET(std::list) explode(const Vector &point,int n) const;
  VECTORSET(std::list) explode(const Vector &point) const;

  /**
   * Calculate the distance of two points in the periodic space
   * defined by this box
   */
  double periodicDistanceSquared(const Vector &point1,const Vector &point2) const;

  /**
   * Calculate the distance of two points in the periodic space
   * defined by this box
   */
  double periodicDistance(const Vector &point1,const Vector &point2) const;

  const Conditions_t getConditions();
  void setCondition(int,BoundaryCondition_t);

private:
  Conditions_t conditions;
  Matrix *M;    //!< Defines the layout of the box
  Matrix *Minv; //!< Inverse of M to avoid recomputation
};

#endif /* BOX_HPP_ */
