/*
 * Box.cpp
 *
 *  Created on: Jun 30, 2010
 *      Author: crueger
 */

//#include "Helpers/MemDebug.hpp"

#include "Box.hpp"

#include <cmath>

#include "Matrix.hpp"
#include "vector.hpp"

#include "Helpers/Assert.hpp"

Box::Box()
{
  M= new Matrix();
  M->one();
  Minv = new Matrix();
  Minv->one();
  conditions.resize(3);
  conditions[0] = conditions[1] = conditions[2] = Wrap;
}

Box::Box(const Box& src){
  M=new Matrix(*src.M);
  Minv = new Matrix(*src.Minv);
  conditions = src.conditions;
}

Box::~Box()
{
  delete M;
  delete Minv;
}

const Matrix &Box::getM() const{
  return *M;
}
const Matrix &Box::getMinv() const{
  return *Minv;
}

void Box::setM(Matrix _M){
  ASSERT(_M.determinant()!=0,"Matrix in Box construction was not invertible");
  *M    =_M;
  *Minv = M->invert();
}

Vector Box::translateIn(const Vector &point) const{
  return (*M) * point;
}

Vector Box::translateOut(const Vector &point) const{
  return (*Minv) * point;
}

Vector Box::WrapPeriodically(const Vector &point) const{
  Vector helper = translateOut(point);
  for(int i=NDIM;i--;){
    double intpart,fracpart;
    fracpart = modf(helper.at(i),&intpart);
    if(fracpart<0.)
      fracpart+=1.;
    helper.at(i)=fracpart;
  }
  return translateIn(helper);
}

bool Box::isInside(const Vector &point) const
{
  bool result = true;
  Vector tester = (*Minv) * point;

  for(int i=0;i<NDIM;i++)
    result = result && ((tester[i] >= -MYEPSILON) && ((tester[i] - 1.) < MYEPSILON));

  return result;
}


VECTORSET(std::list) Box::explode(const Vector &point,int n) const{
  VECTORSET(std::list) res;

  // translate the Vector into each of the 27 neighbourhoods

  // first create all translation Vectors
  // there are (n*2+1)^3 such vectors
  int max_dim = (n*2+1);
  int max_dim2 = max_dim*max_dim;
  int max = max_dim2*max_dim;
  // only one loop to avoid unneccessary jumps
  for(int i = 0;i<max;++i){
    // get all coordinates for this iteration
    int n1 = (i%max_dim)-n;
    int n2 = ((i/max_dim)%max_dim)-n;
    int n3 = ((i/max_dim2))-n;
    Vector translation = translateIn(Vector(n1,n2,n3));
    res.push_back(translation);
  }
  // translate all the translation vector by the offset defined by point
  res.translate(point);
  return res;
}

VECTORSET(std::list) Box::explode(const Vector &point) const{
  VECTORSET(std::list) res;

  // translate the Vector into each of the 27 neighbourhoods

  // first create all 27 translation Vectors
  // these loops depend on fixed parameters and can easily be expanded
  // by the compiler to allow code without jumps
  for(int n1 = -1;n1<=1;++n1){
    for(int n2 = -1;n2<=1;++n2){
      for(int n3 = -1;n3<=1;++n3){
        // get all coordinates for this iteration
        Vector translation = translateIn(Vector(n1,n2,n3));
        res.push_back(translation);
      }
    }
  }
  // translate all the translation vector by the offset defined by point
  res.translate(point);
  return res;
}

double Box::periodicDistanceSquared(const Vector &point1,const Vector &point2) const{
  VECTORSET(std::list) expansion = explode(point1);
  double res = expansion.minDistSquared(point2);
  return res;
}

double Box::periodicDistance(const Vector &point1,const Vector &point2) const{
  double res;
  res = sqrt(periodicDistanceSquared(point1,point2));
  return res;
}

const Box::Conditions_t Box::getConditions(){
  return conditions;
}

void Box::setCondition(int i,Box::BoundaryCondition_t condition){
  conditions[i]=condition;
}

Box &Box::operator=(const Box &src){
  if(&src!=this){
    delete M;
    delete Minv;
    M    = new Matrix(*src.M);
    Minv = new Matrix(*src.Minv);
    conditions = src.conditions;
  }
  return *this;
}

Box &Box::operator=(const Matrix &mat){
  setM(mat);
  return *this;
}
