/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2012 University of Bonn. All rights reserved.
 * Please see the COPYING file or "Copyright notice" in builder.cpp for details.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>. 
 */

/*
 * HomologyGraph.cpp
 *
 *  Created on: Sep 24, 2012
 *      Author: heber
 */


// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// include headers that implement a archive in simple text format
// otherwise BOOST_CLASS_EXPORT_IMPLEMENT has no effect
#include <boost/archive/text_oarchive.hpp>
#include <boost/archive/text_iarchive.hpp>

#include "CodePatterns/MemDebug.hpp"

#include "HomologyGraph.hpp"

#include <iostream>

HomologyGraph::HomologyGraph(const KeySet &keyset) :
  nodes(detail::getNodesFromKeySet(keyset)),
  edges(detail::getEdgesFromKeySet(keyset))
{}

HomologyGraph::HomologyGraph(const IndexSet &index) :
  nodes(detail::getNodesFromIndexSet(index)),
  edges(detail::getEdgesFromIndexSet(index))
{}

bool HomologyGraph::operator<(const HomologyGraph &graph) const
{
  if (nodes < graph.nodes) {
    return true;
  } else if (nodes > graph.nodes) {
    return false;
  } else {
    if (edges < graph.edges)
      return true;
    else
      return false;
  }
}

bool HomologyGraph::operator>(const HomologyGraph &graph) const
{
  if (nodes > graph.nodes) {
    return true;
  } else if (nodes < graph.nodes) {
    return false;
  } else {
    if (edges > graph.edges)
      return true;
    else
      return false;
  }
}

bool HomologyGraph::operator==(const HomologyGraph &graph) const
{
  if (nodes != graph.nodes) {
    return false;
  } else {
    return (edges == graph.edges);
  }
}

HomologyGraph& HomologyGraph::operator=(const HomologyGraph &graph)
{
  // self-assignment check
  if (this != &graph) {
    const_cast<nodes_t &>(nodes) = graph.nodes;
    const_cast<edges_t &>(edges) = graph.edges;
  }
  return *this;
}

std::ostream& operator<<(std::ostream& ost, const HomologyGraph &graph)
{
  for (HomologyGraph::nodes_t::const_iterator nodeiter = graph.nodes.begin();
      nodeiter != graph.nodes.end();
      ++nodeiter)
    ost << *nodeiter << " ";
  for (HomologyGraph::edges_t::const_iterator edgeiter = graph.edges.begin();
      edgeiter != graph.edges.end();
      ++edgeiter)
    ost << *edgeiter << " ";
  return ost;
}

// we need to explicitly instantiate the serialization functions
BOOST_CLASS_EXPORT_IMPLEMENT(HomologyGraph)

