/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * Chronos.cpp
 *
 *  Created on: Mar 14, 2011
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include <iostream>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#ifdef HAVE_SYS_TIMES_H
# include <sys/times.h>
#else
# include <time.h>
#endif
#else
# include <time.h>
#endif


#include "CodePatterns/Chronos.hpp"

#include "CodePatterns/Singleton_impl.hpp"

Chronos::Chronos()
{}

Chronos::~Chronos()
{}

double Chronos::getTime(const std::string _name) const
{
  // only those functions have a time that have run already
  if (TimeRunning.count(_name)) {
    // return -1 if function is currently running
    if (TimeRunning.count(_name) != 0.)
      return AccountedTime.at(_name);
    else
      return -1.;
  }
  return 0.;
}

void Chronos::resetTime(const std::string _name)
{
  if (TimeRunning.count(_name)) {
    AccountedTime[_name] = 0.;
  }
}

void Chronos::startTiming(const std::string _name)
{
  // start time keeping
  TimeRunning[_name] = getCurrentTime();
}

double Chronos::getCurrentTime() const
{
#ifdef HAVE_SYS_TIMES_H
  struct tms *buffer = new tms;
  double currenttime;
  if (times(buffer) != (clock_t)(-1))
    currenttime = ((double)buffer->tms_utime/(double)sysconf(_SC_CLK_TCK));
  else
    currenttime = 0.;
  delete buffer;
#else
  const double currenttime = (clock()/(double)CLOCKS_PER_SEC);
#endif
  //std::cout << "Current time is " << currenttime << std::endl;
  return currenttime;
}

void Chronos::endTiming(const std::string _name)
{
  const double endtime = getCurrentTime();
  const double starttime = TimeRunning[_name];

  // if present
  ASSERT(TimeRunning.count(_name), "Chronos::endTiming() - no timer under "+_name+" running.");
  // finish time keeping
  const double RunTime = ((double)endtime - starttime);
  TimekeepingMap::iterator iter = AccountedTime.find(_name);
  if (iter != AccountedTime.end())
    AccountedTime[_name] += RunTime;
  else
    AccountedTime[_name] = RunTime;

  // and zero for next run
  TimeRunning[_name] = 0.;
}

double Chronos::SumUpTotalTime() const
{
  double sum = 0.;
  for (TimekeepingMap::const_iterator iter = AccountedTime.begin();
      iter != AccountedTime.end();
      ++iter) {
    sum += iter->second;
  }
  return sum;
}

size_t Chronos::SumUpTotalFunctions() const
{
  return TimeRunning.size();
}

std::ostream& operator<<(std::ostream &ost, const Chronos &_time)
{
  ost << "List of functions present:" << std::endl;
  for (Chronos::TimekeepingMap::const_iterator iter = _time.AccountedTime.begin();
      iter != _time.AccountedTime.end();
      ++iter)
    ost << "\t" << iter->first << "\t" << iter->second << "s" << std::endl;
  ost << "Total time passed: " << _time.SumUpTotalTime() << std::endl;
  ost << "Total functions: " << _time.SumUpTotalFunctions() << std::endl;
  return ost;
}

// construct the remainder of the singleton
CONSTRUCT_SINGLETON(Chronos)

// catch if someone wants to use Info objects in here
#ifdef INFO_HPP_
BOOST_PP_ASSERT_MSG(1,\
  ERROR: This is a safety measure to generate a compiler warning\n \
  if you really try to use info.hpp in __FILE__.)
#endif

