/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * GLMoleculeObject_bond.cpp
 *
 *  Created on: Aug 17, 2011
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "GLMoleculeObject_bond.hpp"

#include <Qt3D/qglbuilder.h>
#include <Qt3D/qglcylinder.h>
#include <Qt3D/qglmaterial.h>
#include <Qt3D/qglscenenode.h>

#include "CodePatterns/MemDebug.hpp"

#include <cmath>

#include "CodePatterns/Assert.hpp"
#include "atom.hpp"
#include "Bond/bond.hpp"
#include "element.hpp"
#include "Helpers/defs.hpp"
#include "LinearAlgebra/Line.hpp"
#include "LinearAlgebra/Vector.hpp"

static QGLSceneNode *createBond(QObject *parent, double distance)
{
   QGLBuilder builder;
   QGLCylinder cylinder(.25,.25,.25);
   cylinder.setHeight(distance);
   builder << cylinder;
   QGLSceneNode *n = builder.finalizedSceneNode();
   n->setParent(parent);
   return n;
}

GLMoleculeObject_bond::GLMoleculeObject_bond(QObject *parent, const bond *bondref, double distance, enum SideOfBond side) :
  GLMoleculeObject(createBond(parent, distance), parent), _bond(bondref)
{
  Vector Position;
  Vector OtherPosition;
  size_t elementno = 0;
  switch (side) {
    case left:
      Position = _bond->leftatom->getPosition();
      OtherPosition = _bond->rightatom->getPosition();
      if (_bond->leftatom->getType() != NULL) {
        elementno = _bond->leftatom->getType()->getNumber();
      } else { // if not element yet set, set to hydrogen
        elementno = 1;
      }
      break;
    case right:
      Position = _bond->rightatom->getPosition();
      OtherPosition = _bond->leftatom->getPosition();
      if (_bond->rightatom->getType() != NULL) {
        elementno = _bond->rightatom->getType()->getNumber();
      } else { // if not element yet set, set to hydrogen
        elementno = 1;
      }

      break;
    default:
      ASSERT(0,
          "GLMoleculeObject_bond::GLMoleculeObject_bond() - side is not a valid argument: "+toString(side)+".");
      break;
  }

  QGLMaterial *elementmaterial = getMaterial(elementno);
  setMaterial(elementmaterial);

  // calculate position
  Vector Z(0.,0.,1.);
  Vector zeroVec(0.,0.,0.);
  Vector a,b;
  Vector OtherAxis;
  double alpha;
  a = Position - OtherPosition;
  // construct rotation axis
  b = a;
  b.VectorProduct(Z);
  Line axis(zeroVec, b);
  // calculate rotation angle
  alpha = a.Angle(Z);
  // construct other axis to check right-hand rule
  OtherAxis = b;
  OtherAxis.VectorProduct(Z);
  // assure right-hand rule for the rotation
  if (a.ScalarProduct(OtherAxis) < MYEPSILON)
    alpha = M_PI-alpha;
  // check
  Vector a_rotated = axis.rotateVector(a, alpha);
  std::cout << "Created cylinder from "// << Position << " to " << OtherPosition
      << a << " to " << a_rotated << " around " << b << " by " << alpha/M_PI*180. << ", respectively." << endl;

  // set position
  setPosition(QVector3D(Position[0], Position[1], Position[2]));
  setRotationVector(QVector3D(b[0], b[1], b[2]));
  setRotationAngle(alpha/M_PI*180.);
}
