/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * GLWorldScene.cpp
 *
 *  This is based on the Qt3D example "teaservice", specifically parts of teaservice.cpp.
 *
 *  Created on: Aug 17, 2011
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "GLWorldScene.hpp"

#include "GLMoleculeObject.hpp"
#include "GLMoleculeObject_atom.hpp"
#include "GLMoleculeObject_bond.hpp"

#include "CodePatterns/MemDebug.hpp"

#include "CodePatterns/Log.hpp"

#include "atom.hpp"
#include "Bond/bond.hpp"
#include "molecule.hpp"
#include "World.hpp"


GLWorldScene::GLWorldScene(QObject *parent)
   : QObject(parent)
{
  init();

  //changeMaterials(false);
}

GLWorldScene::~GLWorldScene()
{
  // remove all elements
  GLMoleculeObject::cleanMaterialMap();
}

/** Initialise the WorldScene with molecules and atoms from World.
 *
 */
void GLWorldScene::init()
{
  const std::vector<molecule*> &molecules = World::getInstance().getAllMolecules();

  if (molecules.size() > 0) {
    for (std::vector<molecule*>::const_iterator Runner = molecules.begin();
        Runner != molecules.end();
        Runner++) {
      // create molecule
      for (molecule::const_iterator atomiter = (*Runner)->begin();
          atomiter != (*Runner)->end();
          ++atomiter) {
        // create atom objects in scene
        atomInserted(*atomiter);

        // create its bonds
        const BondList &bonds = (*atomiter)->getListOfBonds();
        for (BondList::const_iterator bonditer = bonds.begin();
            bonditer != bonds.end();
            ++bonditer) {
          if ((*bonditer)->leftatom->getId() == (*atomiter)->getId()) {
            // create bond objects in scene
            bondInserted(*bonditer);
          }
        }
      }
    }
  }
}

/** Adds an atom to the scene.
 *
 * @param _atom atom to add
 */
void GLWorldScene::atomInserted(const atom *_atom)
{
  LOG(0, "GLWorldScene: Received signal atomInserted for atom "+toString(_atom->getId())+".");
  GLMoleculeObject_atom *atomObject = new GLMoleculeObject_atom(this, _atom);
  AtomNodeMap::iterator iter = AtomsinMoleculeMap.find(_atom->getId());
  ASSERT(iter == AtomsinMoleculeMap.end(),
      "GLWorldScene::atomAdded() - same atom "+_atom->getName()+" added again.");
  AtomsinMoleculeMap.insert( make_pair(_atom->getId(), atomObject) );
  connect (atomObject, SIGNAL(clicked(atomId_t)), this, SLOT(atomClicked(atomId_t)));
  connect (atomObject, SIGNAL(hoverChanged()), this, SIGNAL(hoverChanged()));
  emit changed();
}

/** Removes an atom from the scene.
 *
 * @param _atom atom to remove
 */
void GLWorldScene::atomRemoved(const atom *_atom)
{
  LOG(0, "GLWorldScene: Received signal atomRemoved for atom "+toString(_atom->getId())+".");
  // remove all its bonds
  const BondList& bondlist = _atom->getListOfBonds();
  for (BondList::const_iterator iter = bondlist.begin(); iter != bondlist.end(); ++iter) {
    bondRemoved(*iter);
  }
  // remove atoms
  AtomNodeMap::iterator iter = AtomsinMoleculeMap.find(_atom->getId());
  ASSERT(iter != AtomsinMoleculeMap.end(),
      "GLWorldScene::atomRemoved() - atom "+_atom->getName()+" not on display.");
  GLMoleculeObject_atom *atomObject = iter->second;
  atomObject->disconnect();
  AtomsinMoleculeMap.erase(iter);
  delete atomObject;
  emit changed();
}

/** Adds an bond to the scene.
 *
 * @param _bond bond to add
 */
void GLWorldScene::bondInserted(const bond *_bond)
{
  const double distance =
      _bond->leftatom->getPosition().distance(_bond->rightatom->getPosition())/2.;
  {
    // left bond
    const BondIds Leftids( make_pair(_bond->leftatom->getId(), _bond->rightatom->getId()) );
    BondNodeMap::iterator iter = BondsinMoleculeMap.find(Leftids);
    ASSERT(iter == BondsinMoleculeMap.end(),
        "GLWorldScene::bondAdded() - same left-sided bond "+toString(*_bond)+" added again.");
    GLMoleculeObject_bond *bondObject =
        new GLMoleculeObject_bond(this, _bond, distance, GLMoleculeObject_bond::left);
    BondsinMoleculeMap.insert( make_pair(Leftids, bondObject) );
  }
  {
    // right bond
    const BondIds Rightids( make_pair(_bond->rightatom->getId(), _bond->leftatom->getId()) );
    BondNodeMap::iterator iter = BondsinMoleculeMap.find(Rightids);
    ASSERT(iter == BondsinMoleculeMap.end(),
        "GLWorldScene::bondAdded() - same right-sided bond "+toString(*_bond)+" added again.");
    GLMoleculeObject_bond *bondObject =
        new GLMoleculeObject_bond(this, _bond, distance, GLMoleculeObject_bond::right);
    BondsinMoleculeMap.insert( make_pair(Rightids, bondObject) );
  }
  emit changed();
}

/** Removes an bond from the scene.
 *
 * @param _bond bond to remove
 */
void GLWorldScene::bondRemoved(const bond *_bond)
{
  {
    // left bond
    const BondIds Leftids( make_pair(_bond->leftatom->getId(), _bond->rightatom->getId()) );
    BondNodeMap::iterator leftiter = BondsinMoleculeMap.find( Leftids );
    ASSERT(leftiter != BondsinMoleculeMap.end(),
        "GLWorldScene::bondRemoved() - bond "+toString(*_bond)+" not on display.");
    GLMoleculeObject_bond *bondObject = leftiter->second;
    BondsinMoleculeMap.erase(leftiter);
    delete bondObject;
  }
  {
    // right bond
    const BondIds Rightids( make_pair(_bond->rightatom->getId(), _bond->leftatom->getId()) );
    BondNodeMap::iterator rightiter = BondsinMoleculeMap.find( Rightids );
    ASSERT(rightiter != BondsinMoleculeMap.end(),
        "GLWorldScene::bondRemoved() - bond "+toString(*_bond)+" not on display.");
    GLMoleculeObject_bond *bondObject = rightiter->second;
    BondsinMoleculeMap.erase(rightiter);
    delete bondObject;
  }
  emit changed();
}

void GLWorldScene::initialize(QGLView *view, QGLPainter *painter) const
{
  // Initialize all of the mesh objects that we have as children.
   foreach (QObject *obj, children()) {
     GLMoleculeObject *meshobj = qobject_cast<GLMoleculeObject *>(obj);
       if (meshobj)
         meshobj->initialize(view, painter);
   }
}

void GLWorldScene::draw(QGLPainter *painter) const
{
   // Draw all of the mesh objects that we have as children.
   foreach (QObject *obj, children()) {
     GLMoleculeObject *meshobj = qobject_cast<GLMoleculeObject *>(obj);
       if (meshobj)
         meshobj->draw(painter);
   }
}

void GLWorldScene::atomClicked(atomId_t no)
{
   qDebug("GLWorldScene: atom %d has been clicked.", no);
   emit clicked(no);
}

