/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * GLMoleculeObject_molecule.cpp
 *
 *  Created on: Mar 30, 2012
 *      Author: ankele
 */





// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "GLMoleculeObject_molecule.hpp"

#include <Qt3D/qglscenenode.h>

#include "CodePatterns/MemDebug.hpp"

#include "CodePatterns/Assert.hpp"
#include "CodePatterns/Log.hpp"
#include "CodePatterns/Observer/Notification.hpp"
#include "CodePatterns/Observer/ObserverLog.hpp"

#include "Atom/atom.hpp"
#include "molecule.hpp"
#include "Descriptors/AtomIdDescriptor.hpp"
#include "Element/element.hpp"
#include "LinearAlgebra/Vector.hpp"
#include "World.hpp"

GLMoleculeObject_molecule::GLMoleculeObject_molecule(QGLSceneNode *mesh[], QObject *parent, const molecule *molref) :
  GLMoleculeObject(mesh, parent),
  Observer(std::string("GLMoleculeObject_molecule")+toString(molref->getId())),
  _molecule(molref)
{
  // sign on as observer (obtain non-const instance before)
  /*molref->signOn(this, AtomObservable::IndexChanged);
  molref->signOn(this, AtomObservable::PositionChanged);
  molref->signOn(this, AtomObservable::ElementChanged);
  molref->signOn(this, AtomObservable::BondsAdded);*/
  setMaterial(getMaterial(1));
  World::getInstance().signOn(this, World::SelectionChanged);
  updateBoundingBox();
}

GLMoleculeObject_molecule::~GLMoleculeObject_molecule()
{
  /*_atom->signOff(this, AtomObservable::IndexChanged);
  _atom->signOff(this, AtomObservable::PositionChanged);
  _atom->signOff(this, AtomObservable::ElementChanged);
  _atom->signOff(this, AtomObservable::BondsAdded);*/
  World::getInstance().signOff(this, World::SelectionChanged);
}

void GLMoleculeObject_molecule::updateBoundingBox()
{
  Shape shape = _molecule->getBoundingShape();
  Vector v = shape.getCenter();
  setPosition(QVector3D(v[0], v[1], v[2]));
  setScale(shape.getRadius() + 0.3); // getBoundingShape() only sees atoms as points, so make the box a bit bigger
}

void GLMoleculeObject_molecule::update(Observable *publisher)
{
  const molecule *_mol = static_cast<molecule *>(publisher);
#ifdef LOG_OBSERVER
  observerLog().addMessage() << "++ Update of Observer " << observerLog().getName(static_cast<Observer *>(this)) << " from molecule "+toString(_molecule->getId())+".";
#endif
}

void GLMoleculeObject_molecule::subjectKilled(Observable *publisher)
{}

void GLMoleculeObject_molecule::recieveNotification(Observable *publisher, Notification_ptr notification)
{
  if (publisher == dynamic_cast<const Observable*>(_molecule)){
    // notofication from atom
#ifdef LOG_OBSERVER
    observerLog().addMessage() << "++ Update of Observer "<< observerLog().getName(static_cast<Observer *>(this))
          << " received notification from molecule " << _molecule->getId() << " for channel "
          << notification->getChannelNo() << ".";
#endif
  }else{
    // notification from world
#ifdef LOG_OBSERVER
    observerLog().addMessage() << "++ Update of Observer "<< observerLog().getName(static_cast<Observer *>(this))
          << " received notification from world for channel "
          << notification->getChannelNo() << ".";
#endif
    switch (notification->getChannelNo()) {
      case World::SelectionChanged:
        setSelected(World::getInstance().isSelected(_molecule));
        break;
      default:
        break;
    }
  }
}

