/*
 * SubgraphDissectionAction.cpp
 *
 *  Created on: May 9, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Helpers/MemDebug.hpp"

#include "Actions/FragmentationAction/SubgraphDissectionAction.hpp"
#include "Actions/ActionRegistry.hpp"
#include "Descriptors/MoleculeDescriptor.hpp"

#include "atom.hpp"
#include "bond.hpp"
#include "bondgraph.hpp"
#include "config.hpp"
#include "Helpers/Log.hpp"
#include "Helpers/Verbose.hpp"
#include "molecule.hpp"
#include "stackclass.hpp"
#include "World.hpp"

#include <iostream>
#include <string>

using namespace std;

#include "UIElements/UIFactory.hpp"
#include "UIElements/Dialog.hpp"
#include "Actions/ValueStorage.hpp"

const char FragmentationSubgraphDissectionAction::NAME[] = "subgraph-dissect";

FragmentationSubgraphDissectionAction::FragmentationSubgraphDissectionAction() :
  Action(NAME)
{}

FragmentationSubgraphDissectionAction::~FragmentationSubgraphDissectionAction()
{}

/** Dissects given \a *mol into connected subgraphs and inserts them as new molecules but with old atoms into \a this.
 */
void FragmentationSubgraphDissection() {
  ActionRegistry::getInstance().getActionByName(FragmentationSubgraphDissectionAction::NAME)->call(Action::NonInteractive);
};

Dialog* FragmentationSubgraphDissectionAction::fillDialog(Dialog *dialog) {
  ASSERT(dialog,"No Dialog given when filling action dialog");

  dialog->queryEmpty(NAME, MapOfActions::getInstance().getDescription(NAME));

  return dialog;
}


Action::state_ptr FragmentationSubgraphDissectionAction::performCall() {
  DoLog(1) && (Log() << Verbose(1) << "Dissecting molecular system into a set of disconnected subgraphs ... " << endl);

  MoleculeListClass *molecules = World::getInstance().getMolecules();
  config * const configuration = World::getInstance().getConfig();

  // 0a. remove all present molecules
  vector<molecule *> allmolecules = World::getInstance().getAllMolecules();
  for (vector<molecule *>::iterator MolRunner = allmolecules.begin(); MolRunner != allmolecules.end(); ++MolRunner) {
    molecules->erase(*MolRunner);
    World::getInstance().destroyMolecule(*MolRunner);
  }

  // 0b. remove all bonds and construct a molecule with all atoms
  molecule *mol = World::getInstance().createMolecule();
  vector <atom *> allatoms = World::getInstance().getAllAtoms();
  for(vector<atom *>::iterator AtomRunner = allatoms.begin(); AtomRunner != allatoms.end(); ++AtomRunner) {
    for(BondList::iterator BondRunner = (*AtomRunner)->ListOfBonds.begin(); !(*AtomRunner)->ListOfBonds.empty(); BondRunner = (*AtomRunner)->ListOfBonds.begin())
      delete(*BondRunner);
    mol->AddAtom(*AtomRunner);
  }

  // 1. create the bond structure of the single molecule
  if (configuration->BG != NULL) {
    if (!configuration->BG->ConstructBondGraph(mol)) {
      World::getInstance().destroyMolecule(mol);
      DoeLog(1) && (eLog()<< Verbose(1) << "There are no bonds." << endl);
      return Action::failure;
    }
  } else {
    DoeLog(1) && (eLog()<< Verbose(1) << "There is no BondGraph class present to create bonds." << endl);
    return Action::failure;
  }

  // 2. scan for connected subgraphs
  MoleculeLeafClass *Subgraphs = NULL;      // list of subgraphs from DFS analysis
  class StackClass<bond *> *BackEdgeStack = NULL;
  Subgraphs = mol->DepthFirstSearchAnalysis(BackEdgeStack);
  delete(BackEdgeStack);
  if ((Subgraphs == NULL) || (Subgraphs->next == NULL)) {
    World::getInstance().destroyMolecule(mol);
    DoeLog(1) && (eLog()<< Verbose(1) << "There are no atoms." << endl);
    return Action::failure;
  }
  int FragmentCounter = Subgraphs->next->Count();

  // TODO: When DepthFirstSearchAnalysis does not use AddCopyAtom() anymore, we don't need to delete all original atoms
  // 3. destroy the original molecule
  for (molecule::iterator AtomRunner = mol->begin(); !mol->empty(); AtomRunner = mol->begin())
    World::getInstance().destroyAtom(*AtomRunner);
  World::getInstance().destroyMolecule(mol);

  // 4. free Leafs
  MoleculeLeafClass *MolecularWalker = Subgraphs;
  while (MolecularWalker->next != NULL) {
    MolecularWalker->Leaf = NULL;
    MolecularWalker = MolecularWalker->next;
    delete(MolecularWalker->previous);
  }
  delete(MolecularWalker);
  DoLog(1) && (Log() << Verbose(1) << "I scanned " << FragmentCounter << " molecules." << endl);

  return Action::success;
}

Action::state_ptr FragmentationSubgraphDissectionAction::performUndo(Action::state_ptr _state) {
//  ParserLoadXyzState *state = assert_cast<ParserLoadXyzState*>(_state.get());

  return Action::failure;
//  string newName = state->mol->getName();
//  state->mol->setName(state->lastName);
//
//  return Action::state_ptr(new ParserLoadXyzState(state->mol,newName));
}

Action::state_ptr FragmentationSubgraphDissectionAction::performRedo(Action::state_ptr _state){
  return Action::failure;
}

bool FragmentationSubgraphDissectionAction::canUndo() {
  return false;
}

bool FragmentationSubgraphDissectionAction::shouldUndo() {
  return false;
}

const string FragmentationSubgraphDissectionAction::getName() {
  return NAME;
}
