/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * VectorContentUnittest.cpp
 *
 *  Created on: Jan 8, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

using namespace std;

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

// include headers that implement a archive in simple text format
#include <boost/archive/text_oarchive.hpp>
#include <boost/archive/text_iarchive.hpp>

#include "VectorContentUnitTest.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( VectorContentTest );


void VectorContentTest::setUp()
{
  v = new VectorContent(3);
};

void VectorContentTest::tearDown()
{
  delete(v);
};

/** Unit test for accessing elements in the vector.
 *
 */
void VectorContentTest::AccessTest()
{
  // set
  for (int i=0;i<3;i++)
    v->at(i) = (double)i;

  // and check
  double *ptr = NULL;
  for (int i=0;i<3;i++) {
    CPPUNIT_ASSERT_EQUAL( (double)i, v->at(i) );
    ptr = v->Pointer(i);
    CPPUNIT_ASSERT_EQUAL( (double)i, *ptr );
  }

  // out of bounds
  //CPPUNIT_ASSERT_EQUAL(0., v->at(3) );
  //CPPUNIT_ASSERT_EQUAL(0., v->at(-1) );
};


/** Unit test for initializing the vector.
 *
 */
void VectorContentTest::InitializaionTest()
{
  // set zero
  v->setZero();
  for (int i=0;i<3;i++)
    CPPUNIT_ASSERT_EQUAL( 0., v->at(i) );

  // set basis
  for (int i=0;i<3;i++) {
    v->setBasis(i);
    for (int j=0;j<3;j++)
      CPPUNIT_ASSERT_EQUAL( i == j ? 1. : 0. , v->at(j) );
  }

  // set all
  v->setValue(1.5);
  for (int i=0;i<3;i++)
    CPPUNIT_ASSERT_EQUAL( 1.5, v->at(i) );
};

/** Unit test for copying vectors.
 *
 */
void VectorContentTest::CopyTest()
{
  // set basis
  VectorContent *dest = NULL;
  for (int i=0;i<3;i++) {
    v->setBasis(i);
    dest = new VectorContent(v);
    for (int j=0;j<3;j++)
      CPPUNIT_ASSERT_EQUAL( v->at(j) , dest->at(j) );

    delete(dest);
  }
};

/** Unit test for exchanging elements of a vector.
 *
 */
void VectorContentTest::ExchangeTest()
{
  // set basis
  for (int i=0;i<3;i++) {
    v->setBasis(i);
    v->SwapElements((i)%3,(i+1)%3);
    for (int j=0;j<3;j++)
      CPPUNIT_ASSERT_EQUAL( (i+1)%3 == j ? 1. : 0. , v->at(j) );
  }

  // set to 1,2,3 and reverse
  for (int i=0;i<3;i++)
    v->at(i) = (double)(i+1);
  v->Reverse();
  for (int j=0;j<3;j++)
    CPPUNIT_ASSERT_EQUAL( (double)(3-j), v->at(j) );
};

/** UnitTest for operators.
 */
void VectorContentTest::OperatorIsTest()
{
  VectorContent zero(3);
  VectorContent unit(3);
  zero.setZero();
  unit.setZero();
  unit[1] = 1.;
  // summation and scaling
  CPPUNIT_ASSERT_EQUAL( true, unit.IsOne() );
  CPPUNIT_ASSERT_EQUAL( false, zero.IsOne() );
  CPPUNIT_ASSERT_EQUAL( false, unit.IsZero() );
  CPPUNIT_ASSERT_EQUAL( true, zero.IsZero() );
};

/** UnitTest for operators.
 */
void VectorContentTest::OperatorAlgebraTest()
{
  VectorContent zero(3);
  VectorContent unit(3);
  zero.setZero();
  unit.setZero();
  unit[1] = 1.;
  // summation and scaling
  CPPUNIT_ASSERT_EQUAL( true, (zero+unit).IsOne() );
  CPPUNIT_ASSERT_EQUAL( true, (zero+unit).IsOne() );
  CPPUNIT_ASSERT_EQUAL( true, (zero-unit).IsOne() );
  CPPUNIT_ASSERT_EQUAL( false, (zero-unit).IsZero() );
  CPPUNIT_ASSERT_EQUAL( true, (zero+zero).IsZero() );
  CPPUNIT_ASSERT_EQUAL( false, (unit*0.98).IsOne() );
  CPPUNIT_ASSERT_EQUAL( false, (0.98*unit).IsOne() );
  CPPUNIT_ASSERT_EQUAL( true, (unit*1.).IsOne() );
  CPPUNIT_ASSERT_EQUAL( true, (1.*unit).IsOne() );

  CPPUNIT_ASSERT_EQUAL( unit, (zero+unit) );
  CPPUNIT_ASSERT_EQUAL( zero, (zero+zero) );
  CPPUNIT_ASSERT_EQUAL( unit, (unit+zero) );

  unit += zero;
  CPPUNIT_ASSERT_EQUAL( true, unit.IsOne() );
  unit *= 1.;
  CPPUNIT_ASSERT_EQUAL( true, unit.IsOne() );
};

/** Unit test for reading from and writing vector to stream
 *
 */
void VectorContentTest::ReadWriteTest()
{
  // set up matrix
  VectorContent v((size_t)3);
  for (size_t i=0; i<3;++i)
    v.at(i) = i+1.;

  // write to stream
  std::stringstream vectorstream;
  v.write(vectorstream);

  // parse in dimensions and check
  size_t vectordimension = VectorContent::preparseVectorDimensions(vectorstream);
  CPPUNIT_ASSERT_EQUAL( (size_t)3, vectordimension );
  // parse in vector and check
  VectorContent w(3, vectorstream);
  CPPUNIT_ASSERT_EQUAL( v, w );
}

/** UnitTest for serialization.
 */
void VectorContentTest::SerializationTest()
{
  v->setZero();
  (*v)[0] = 0.;
  (*v)[1] = 1.;
  (*v)[2] = 2.;
  // write element to stream
  std::stringstream stream;
  boost::archive::text_oarchive oa(stream);
  oa << v;

  //std::cout << "Contents of archive is " << stream.str() << std::endl;

  // create and open an archive for input
  boost::archive::text_iarchive ia(stream);
  // read class state from archive
  VectorContent *newv;

  ia >> newv;

  CPPUNIT_ASSERT (*v == *newv);
}
