/*
 * QtMoleculeItem.hpp
 *
 *  Created on: Jan 17, 2015
 *      Author: heber
 */

#ifndef QTMOLECULEITEM_HPP_
#define QTMOLECULEITEM_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <QList>
#include <QStandardItem>

#include "CodePatterns/Observer/Observer.hpp"

#include <boost/function.hpp>
#include <list>
#include <string>

#include "molecule.hpp"

/** This class describes the general interface for a specific item in a QtAbstractItemModel
 * to contain a specific piece of information about a specific molecule.
 *
 * To this end, we need a ref to the molecule for updating the information when
 * necessary and we need to know the channels we have to listen to know when an
 * update is necessary.
 *
 * This class takes care of of all the rest:
 * -# informing QtMoleculeList about required update ("dirty")
 * -# relaying updateState() call from QtMoleculeList to the specific internal_updateState()
 */
class QtMoleculeItem : public QStandardItem, public Observer
{
public:
  //!> typedef for a list of notification channels required, to know when to update
  typedef std::list<molecule::NotificationType> channellist_t;

  //!> enum states whether an item's formula changed or not
  enum MoveTypes {
    NeedsMove,
    DoesNotMove
  };

  //!> typedef for callback function to model to inform when we need update
  typedef const boost::function<void (QtMoleculeItem*, enum MoveTypes)> emitDirtyState_t;

  QtMoleculeItem(
      const molecule *_mol,
      const channellist_t &_channellist,
      const enum MoveTypes _movetype,
      const emitDirtyState_t _emitDirtyState);
  virtual ~QtMoleculeItem();

  /** Update the state of this item.
   *
   */
  void updateState();

private:
  void update(Observable *publisher);
  void recieveNotification(Observable *publisher, Notification_ptr notification);
  void subjectKilled(Observable *publisher);

  void signOnToMolecule();
  void signOffFromMolecule();

protected:
  /** This function needs to be implemented to make the specific update.
   *
   */
  virtual void internal_updateState() = 0;

  //!> molecule we are signed on to
  const molecule *mol;

private:
  //!> move type for this specific item, set this when implementing
  const enum MoveTypes movetype;

  //!> ref to the specific static list of channels we are signed on to
  const channellist_t &channellist;

  //!> states whether we are still signed on to molecule
  bool IsSignedOn;

  //!> states that this item needs to be updated
  bool dirty;

  //!> bound callback function to inform model about change
  const emitDirtyState_t emitDirtyState;
};


#endif /* QTMOLECULEITEM_HPP_ */
