/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2012 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * \file poolworker.cpp
 *
 * This file strongly follows the Serialization example from the boost::asio
 * library (see client.cpp)
 *
 *  Created on: Feb 28, 2012
 *      Author: heber
 */


// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// boost asio needs specific operator new
#include <boost/asio.hpp>

#include "CodePatterns/MemDebug.hpp"

#include <iostream>
#include <boost/program_options.hpp>
#include <boost/lexical_cast.hpp>
#include <signal.h>
#include <vector>

#include "Fragmentation/Automation/atexit.hpp"
#include "CodePatterns/Info.hpp"
#include "CodePatterns/Log.hpp"
#include "Pool/PoolWorker.hpp"
#include "atexit.hpp"
#include "Jobs/SystemCommandJob.hpp"
#include "WorkerOptions.hpp"

//!> global shutdown function
boost::function<void (int)> shutdownfunction;
//!> global signal vector for later releasing
std::vector<size_t> signals;
//!> counter for the number of received signals
size_t NoSignalsReceived = 0;
//!> maximum number of received signals after which the handlers are released
const size_t MAX_NOSIGNALSRECEIVED = 3;

void signalhandler(int sig)
{
  // increment received signal counter
  ++NoSignalsReceived;

  // shutdown if we have handler
  if (shutdownfunction)
    shutdownfunction(sig);

  if (NoSignalsReceived >= MAX_NOSIGNALSRECEIVED) {
    // release signal hook again
    for (std::vector<size_t>::const_iterator iter = signals.begin();
        iter != signals.end(); ++iter)
      signal(*iter, NULL);
  }
}


int main(int argc, char* argv[])
{
  // from this moment on, we need to be sure to deeinitialize in the correct order
  // this is handled by the cleanup function
  atexit(cleanUp);

  WorkerOptions WorkerOpts;
  boost::program_options::variables_map vm;

  // Declare the supported options.
  boost::program_options::options_description desc("Allowed options");
  desc.add_options()
      ("help,h", "produce help message")
      ("verbosity,v", boost::program_options::value<size_t>(), "set verbosity level")
      ("signal", boost::program_options::value< std::vector<size_t> >(), "set signal to catch (can be given multiple times)")
      ("server", boost::program_options::value< std::string>(), "connect to server at this address (host:port)")
      ("listen", boost::program_options::value< std::string >(), "listen on this port")
      ("hostname", boost::program_options::value< std::string>(), "name of host on which this codes runs and which server can resolve")
  ;

  boost::program_options::store(boost::program_options::parse_command_line(argc, argv, desc), vm);
  boost::program_options::notify(vm);

  int status = 0;
  if (status) return status;
  status = WorkerOpts.parseHelp(vm, desc);
  if (status) return status;
  status = WorkerOpts.parseVerbosity(vm);
  if (status) return status;
  status = WorkerOpts.parseServer(vm);
  if (status) return status;
  status = WorkerOpts.parseListenPort(vm);
  if (status) return status;
  status = WorkerOpts.parseLocalhost(vm);
  if (status) return status;
  status = WorkerOpts.parseSignals(vm);
  if (status) return status;

  size_t exitflag = 0;
  try
  {
    boost::asio::io_service io_service;
    PoolWorker client(io_service, WorkerOpts.server, WorkerOpts.serverport, WorkerOpts.hostname, WorkerOpts.listenport);

    // catch ctrl-c and shutdown worker properly
    shutdownfunction = boost::bind(&PoolWorker::shutdown, boost::ref(client), _1);
    for (std::vector<size_t>::const_iterator iter = WorkerOpts.signals.begin();
        iter != WorkerOpts.signals.end(); ++iter) {
      LOG(0, "STATUS: Catching signal " << *iter << " via signal handler.");
      signal(*iter, &signalhandler);
    }

    // process io requests
    {
      Info info("io_service");
      io_service.run();
    }
    exitflag = client.getFlag();
  }
  catch (std::exception& e)
  {
    std::cerr << e.what() << std::endl;
  }


  return exitflag;
}
