/*
 * GLMoleculeObject_molecule.hpp
 *
 *  Created on: Mar 30, 2012
 *      Author: ankele
 */

#ifndef GLMOLECULEOBJECT_MOLECULE_HPP_
#define GLMOLECULEOBJECT_MOLECULE_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "GLMoleculeObject.hpp"

#include <Qt3D/qgeometrydata.h>

#include "CodePatterns/Cacheable.hpp"
#include "CodePatterns/Observer/Observer.hpp"
#include "CodePatterns/ObservedValue.hpp"

#include "GLMoleculeObject_bond.hpp"

class atom;
class bond;
class GLMoleculeObject_atom;
class GLWorldScene;
class molecule;

class GLMoleculeObject_molecule : public GLMoleculeObject, public Observer
{
  Q_OBJECT
public:
  GLMoleculeObject_molecule(QObject *parent, const moleculeId_t molid);
  GLMoleculeObject_molecule(QGLSceneNode *mesh[], QObject *parent, const moleculeId_t molid);
  virtual ~GLMoleculeObject_molecule();

  // Observer functions
  void update(Observable *publisher);
  void subjectKilled(Observable *publisher);
  void recieveNotification(Observable *publisher, Notification_ptr notification);

  void initialize(QGLView *view, QGLPainter *painter);
  void draw(QGLPainter *painter, const QVector4D &cameraPlane);

  typedef std::pair< atomId_t, atomId_t> BondIds;
  friend std::ostream &operator<<(std::ostream &ost, const BondIds &t);

  static BondIds getBondIds(
      const bond::ptr _bond,
      const enum GLMoleculeObject_bond::SideOfBond side);

signals:
  void changed();
  void changeOccured();
  void hoverChanged(const atomId_t);
  void hoverChanged(const moleculeId_t, int);
  void indexChanged(GLMoleculeObject_molecule *, const moleculeId_t, const moleculeId_t);
  void atomClicked(atomId_t no);
  void moleculeClicked(moleculeId_t no);
  void TesselationHullChanged();
  void BoundingBoxChanged();
  void IsSelectedChanged();
  void AtomInserted(const atomId_t _id);
  void AtomRemoved(const atomId_t _id);
  void IdChanged();

private slots:
  //!> grant GLWorldScene access to private slots
  friend class GLWorldScene;

  void wasClicked();
  void atomInserted(const atomId_t _id);
  void atomRemoved(const atomId_t _id);
  void bondInserted(const atomId_t, const atomId_t, const GLMoleculeObject_bond::SideOfBond side);
  void bondRemoved(const atomId_t leftnr, const atomId_t rightnr);
  void hoverChangedSignalled(GLMoleculeObject *ob);
  void changeAtomId(GLMoleculeObject_atom *, const atomId_t, const atomId_t);

  void setVisible(bool value);

  void activateObserver();
  void deactivateObserver();

  void resetTesselationHull();
  void resetIsSelected();
  void resetBoundingBox();
  void resetAtoms();
  void resetIndex();

private:
  static const molecule * const getMolecule(const moleculeId_t _id);

private:
  void addAtomBonds(
      const bond::ptr &_bond,
      const GLMoleculeObject_bond::SideOfBond _side
      );
  void addAtomBonds(const atom *_atom);

  //!> typedef for the internal set of atoms
  typedef std::set<atomId_t> atoms_t;

  /** Structure for the required information on the bounding box.
   *
   */
  struct BoundingBoxInfo {
    //!> position of center
    Vector position;
    //!> radius of sphere
    double radius;
  };

  /** Structure for the required information on the tesselation hull.
   *
   */
  struct TesselationHullInfo {
  };

  QGeometryData updateTesselationHull() const;
  bool updateIsSelected() const;
  BoundingBoxInfo updateBoundingBox() const;
  atoms_t updateAtoms();
  moleculeId_t updateIndex() const;

  //!> the Observable we are signed on, also indicates whether we are sign on (not NULL)
  const Observable * owner;

  //!> internal variable for caching molecule ref in cstor
  const molecule * const molref;

  //!> list of channels when contained atoms needs to update
  static const Observable::channels_t AtomsChannels;
  //!> list of channels when tesselation hull needs to update
  static const Observable::channels_t HullChannels;
  //!> list of channels when selection status needs to update
  static const Observable::channels_t SelectionStatusChannels;
  //!> list of channels when bounding box needs to update
  static const Observable::channels_t BoundingBoxChannels;
  //!> list of channels when the index needs to update
  static const Observable::channels_t IndexChannels;

  boost::function<moleculeId_t ()> MolIndexUpdater;
  boost::function<QGeometryData ()> TesselationHullUpdater;
  boost::function<bool ()> IsSelectedUpdater;
  boost::function<BoundingBoxInfo ()> BoundingBoxUpdater;
  boost::function<atoms_t ()> PresentAtomsUpdater;

  //!> contains the current molecule index
  ObservedValue<moleculeId_t> MolIndex;
  //!> contains current version of the tesselation hull on request
  Cacheable<QGeometryData> TesselationHull;
  //!> contains newest version of the selection status
  ObservedValue<bool> isSelected;
  //!> contains newest version of the bounding box on request
  ObservedValue<BoundingBoxInfo> BoundingBox;
  //!> contains the current live set of atoms for the molecule
  ObservedValue<atoms_t> PresentAtoms;

  //!> contains the set of atoms displayed
  atoms_t DisplayedAtoms;

  typedef std::map< atomId_t, GLMoleculeObject_atom* > AtomNodeMap;
  typedef std::map< BondIds , GLMoleculeObject_bond* > BondNodeMap;
  AtomNodeMap AtomsinSceneMap;
  BondNodeMap BondsinSceneMap;

  atomId_t hoverAtomId;
};

std::ostream &operator<<(std::ostream &ost, const GLMoleculeObject_molecule::BondIds &t);


#endif /* GLMOLECULEOBJECT_MOLECULE_HPP_ */
