/** \file bond.cpp
 * 
 * Function implementations for the classes BondLeaf, BondTree and bond.
 * 
 */

#include "atom.hpp"
#include "bond.hpp"
#include "element.hpp"
#include "lists.hpp"


/***************************************** Functions for class bond ********************************/

/** Empty Constructor for class bond.
 */
bond::bond() : leftatom(NULL), rightatom(NULL), previous(NULL), next(NULL), HydrogenBond(0), BondDegree(0), nr(-1), Cyclic(false), Type(Undetermined), Used(white)
{
};

/** Constructor for class bond, taking right and left bond partner
 * \param *left left atom
 * \param *right right atom
 * \param degree bond degree
 * \param number increasing index
 */
bond::bond(atom *left, atom *right, const int degree, const int number) : leftatom(left), rightatom(right), previous(NULL), next(NULL), HydrogenBond(0), BondDegree(degree), nr(number), Cyclic(false), Type(Undetermined), Used(white)
{
  if ((left != NULL) && (right != NULL)) {
    if ((left->type != NULL) && (left->type->Z == 1))
      HydrogenBond++;
    if ((right->type != NULL) && (right->type->Z == 1))
      HydrogenBond++;
  }
};

/** Empty Destructor for class bond.
 */
bond::~bond()
{
  // remove this node from the list structure
  if (leftatom != NULL)
    leftatom->UnregisterBond(this);
  if (rightatom != NULL)
  rightatom->UnregisterBond(this);
  unlink(this);
};

ostream & operator << (ostream &ost, const bond &b)
{
  ost << "[" << b.leftatom->getName() << " <" << b.BondDegree << "(H" << b.HydrogenBond << ")>" << b.rightatom->getName() << "]";
  return ost;
};

/** Get the other atom in a bond if one is specified.
 * \param *Atom the pointer to the one atom
 * \return pointer to the other atom in the bond, NULL if no match (indicates something's wrong with the bond) 
 */
atom * bond::GetOtherAtom(const ParticleInfo * const Atom) const
{
  if(leftatom == Atom) 
    return rightatom;
  if(rightatom == Atom) 
    return leftatom;
  DoeLog(1) && (eLog()<< Verbose(1) << "Bond " << *this << " does not contain atom " << *Atom << "!" << endl);
  return NULL;
};


/** Returns whether vertex was used in DFS.
 * \return bond::Used
 */
enum Shading bond::IsUsed() 
{
  return Used;
};

/** Checks if an atom exists in a bond.
 * \param *ptr pointer to atom
 * \return true if it is either bond::leftatom or bond::rightatom, false otherwise
 */
bool bond::Contains(const ParticleInfo * const ptr)
{
  return ((leftatom == ptr) || (rightatom == ptr));
};

/** Checks if an atom exists in a bond.
 * \param nr index of atom
 * \return true if it is either bond::leftatom or bond::rightatom, false otherwise
 */
bool bond::Contains(const int number)
{
  return ((leftatom->nr == number) || (rightatom->nr == number));
};

/** Masks vertex as used in DFS.
 * \return bond::Used, false if bond was already marked used
 */
bool bond::MarkUsed(const enum Shading color) {
  if (Used == black) {
    DoeLog(1) && (eLog()<< Verbose(1) << "Bond " << this << " was already marked black!." << endl);
    return false;
  } else {
    Used = color;
    return true;
  }
};

/** Resets used flag in DFS.
 * \return bond::Used
 */
void bond::ResetUsed() {
  Used = white;
};

/** Calculates the bond length.
 * \return |a - b| with a = bond::leftatom and b = bond::rightatom.
 */
double bond::GetDistance() const
{
  return (leftatom->node->distance(*rightatom->node));
};

/** Calculates the bond length.
 * \return |a - b|^2 with a = bond::leftatom and b = bond::rightatom.
 */
double bond::GetDistanceSquared() const
{
  return (leftatom->node->DistanceSquared(*rightatom->node));
};
