/*
 * analysis_bonds.cpp
 *
 *  Created on: Nov 7, 2009
 *      Author: heber
 */

#include "analysis_bonds.hpp"
#include "atom.hpp"
#include "bond.hpp"
#include "element.hpp"
#include "info.hpp"
#include "log.hpp"
#include "molecule.hpp"

/** Calculates the min, mean and maximum bond counts for the given molecule.
 * \param *mol molecule with atoms and atom::ListOfBonds
 * \param &Min minimum count on return
 * \param &Mean mean count on return
 * \param &Max maximum count on return
 */
void GetMaxMinMeanBondCount(const molecule * const mol, double &Min, double &Mean, double &Max)
{
  Min = 2e+6;
  Max = -2e+5;
  Mean = 0.;

  atom *Walker = mol->start;
  int AtomCount = 0;
  while (Walker->next != mol->end) {
    Walker = Walker->next;
    const int count = Walker->ListOfBonds.size();
    if (Max < count)
      Max = count;
    if (Min > count)
      Min = count;
    Mean += count;
    AtomCount++;
  }
  if (((int)Mean % 2) != 0)
    DoeLog(1) && (eLog()<< Verbose(1) << "Something is wrong with the bond structure, the number of bonds is not even!" << endl);
  Mean /= (double)AtomCount;
};

/** Calculates the min and max bond distance of all atoms of two given elements.
 * \param *mol molecule with atoms
 * \param *type1 one element
 * \param *type2 other element
 * \param &Min minimum distance on return, 0 if no bond between the two elements
 * \param &Mean mean distance (i.e. sum of distance for matching element pairs, divided by number) on return, 0 if no bond between the two elements
 * \param &Max maximum distance on return, 0 if no bond between the two elements
 */
void MinMeanMaxBondDistanceBetweenElements(const molecule *mol, element *type1, element *type2, double &Min, double &Mean, double &Max)
{
  Min = 2e+6;
  Mean = 0.;
  Max = -2e+6;

  int AtomNo = 0;
  atom *Walker = mol->start;
  while (Walker->next != mol->end) {
    Walker = Walker->next;
    if (Walker->type == type1)
      for (BondList::const_iterator BondRunner = Walker->ListOfBonds.begin(); BondRunner != Walker->ListOfBonds.end(); BondRunner++)
        if ((*BondRunner)->GetOtherAtom(Walker)->type == type2) {
          const double distance = (*BondRunner)->GetDistanceSquared();
          if (Min > distance)
            Min = distance;
          if (Max < distance)
            Max = distance;
          Mean += sqrt(distance);
          AtomNo++;
        }
  }
  if (Max < 0) {
    Max = Min = 0.;
  } else {
    Max = sqrt(Max);
    Min = sqrt(Min);
    Mean = Mean/(double)AtomNo;
  }
};

/** Calculate the angle between \a *first and \a *origin and \a *second and \a *origin.
 * \param *first first Vector
 * \param *origin origin of angle taking
 * \param *second second Vector
 * \return angle between \a *first and \a *second, both relative to origin at \a *origin.
 */
double CalculateAngle(Vector *first, Vector *central, Vector *second)
{
  Vector OHBond;
  Vector OOBond;

  OHBond = (*first) - (*central);
  OOBond = (*second) - (*central);
  const double angle = OHBond.Angle(OOBond);
  return angle;
};

/** Checks whether the angle between \a *Oxygen and \a *Hydrogen and \a *Oxygen and \a *OtherOxygen is less than 30 degrees.
 * Note that distance criterion is not checked.
 * \param *Oxygen first oxygen atom, bonded to \a *Hydrogen
 * \param *Hydrogen hydrogen bonded to \a *Oxygen
 * \param *OtherOxygen other oxygen atom
 * \return true - angle criteria fulfilled, false - criteria not fulfilled, angle greater than 30 degrees.
 */
bool CheckHydrogenBridgeBondAngle(atom *Oxygen, atom *Hydrogen, atom *OtherOxygen)
{
  Info FunctionInfo(__func__);

  // check angle
  if (CalculateAngle(&Hydrogen->x, &Oxygen->x, &OtherOxygen->x) < M_PI*(30./180.)) {
    return true;
  } else {
    return false;
  }
};

/** Counts the number of hydrogen bridge bonds.
 * With \a *InterfaceElement an extra element can be specified that identifies some boundary.
 * Then, counting is for the h-bridges that connect to interface only.
 * \param *molecules molecules to count bonds
 * \param *InterfaceElement or NULL
 */
int CountHydrogenBridgeBonds(MoleculeListClass *molecules, element * InterfaceElement = NULL)
{
  atom *Walker = NULL;
  atom *Runner = NULL;
  int count = 0;
  int OtherHydrogens = 0;
  double Otherangle = 0.;
  bool InterfaceFlag = false;
  bool OtherHydrogenFlag = true;
  for (MoleculeList::const_iterator MolWalker = molecules->ListOfMolecules.begin();MolWalker != molecules->ListOfMolecules.end(); MolWalker++) {
    Walker = (*MolWalker)->start;
    while (Walker->next != (*MolWalker)->end) {
      Walker = Walker->next;
      for (MoleculeList::const_iterator MolRunner = molecules->ListOfMolecules.begin();MolRunner != molecules->ListOfMolecules.end(); MolRunner++) {
        Runner = (*MolRunner)->start;
        while (Runner->next != (*MolRunner)->end) {
          Runner = Runner->next;
          if ((Walker->type->Z  == 8) && (Runner->type->Z  == 8)) {
            // check distance
            const double distance = Runner->x.DistanceSquared(Walker->x);
            if ((distance > MYEPSILON) && (distance < HBRIDGEDISTANCE*HBRIDGEDISTANCE)) { // distance >0 means  different atoms
              // on other atom(Runner) we check for bond to interface element and
              // check that O-O line is not in between the shanks of the two connected hydrogens (Otherangle > 104.5)
              OtherHydrogenFlag = true;
              Otherangle = 0.;
              OtherHydrogens = 0;
              InterfaceFlag = (InterfaceElement == NULL);
              for (BondList::const_iterator BondRunner = Runner->ListOfBonds.begin(); BondRunner != Runner->ListOfBonds.end(); BondRunner++) {
                atom * const OtherAtom = (*BondRunner)->GetOtherAtom(Runner);
                // if hydrogen, check angle to be greater(!) than 30 degrees
                if (OtherAtom->type->Z == 1) {
                  const double angle = CalculateAngle(&OtherAtom->x, &Runner->x, &Walker->x);
                  OtherHydrogenFlag = OtherHydrogenFlag && (angle > M_PI*(30./180.) + MYEPSILON);
                  Otherangle += angle;
                  OtherHydrogens++;
                }
                InterfaceFlag = InterfaceFlag || (OtherAtom->type == InterfaceElement);
              }
              DoLog(1) && (Log() << Verbose(1) << "Otherangle is " << Otherangle << " for " << OtherHydrogens << " hydrogens." << endl);
              switch (OtherHydrogens) {
                case 0:
                case 1:
                  break;
                case 2:
                  OtherHydrogenFlag = OtherHydrogenFlag && (Otherangle > M_PI*(104.5/180.) + MYEPSILON);
                  break;
                default: // 3 or more hydrogens ...
                  OtherHydrogenFlag = false;
                  break;
              }
              if (InterfaceFlag && OtherHydrogenFlag) {
                // on this element (Walker) we check for bond to hydrogen, i.e. part of water molecule
                for (BondList::const_iterator BondRunner = Walker->ListOfBonds.begin(); BondRunner != Walker->ListOfBonds.end(); BondRunner++) {
                  atom * const OtherAtom = (*BondRunner)->GetOtherAtom(Walker);
                  if (OtherAtom->type->Z == 1) {
                    // check angle
                    if (CheckHydrogenBridgeBondAngle(Walker, OtherAtom, Runner)) {
                      DoLog(1) && (Log() << Verbose(1) << Walker->getName() << ", " << OtherAtom->getName() << " and " << Runner->getName() << " has a hydrogen bridge bond with distance " << sqrt(distance) << " and angle " << CalculateAngle(&OtherAtom->x, &Walker->x, &Runner->x)*(180./M_PI) << "." << endl);
                      count++;
                      break;
                    }
                  }
                }
              }
            }
          }
        }
      }
    }
  }
  return count;
}

/** Counts the number of bonds between two given elements.
 * \param *molecules list of molecules with all atoms
 * \param *first pointer to first element
 * \param *second pointer to second element
 * \return number of found bonds (\a *first-\a *second)
 */
int CountBondsOfTwo(MoleculeListClass * const molecules, const element * const first, const element * const second)
{
  atom *Walker = NULL;
  int count = 0;

  for (MoleculeList::const_iterator MolWalker = molecules->ListOfMolecules.begin();MolWalker != molecules->ListOfMolecules.end(); MolWalker++) {
    Walker = (*MolWalker)->start;
    while (Walker->next != (*MolWalker)->end) {
      Walker = Walker->next;
      if ((Walker->type == first) || (Walker->type == second)) {  // first element matches
        for (BondList::const_iterator BondRunner = Walker->ListOfBonds.begin(); BondRunner != Walker->ListOfBonds.end(); BondRunner++) {
          atom * const OtherAtom = (*BondRunner)->GetOtherAtom(Walker);
          if (((OtherAtom->type == first) || (OtherAtom->type == second)) && (Walker->nr < OtherAtom->nr)) {
            count++;
            DoLog(1) && (Log() << Verbose(1) << first->name << "-" << second->name << " bond found between " << *Walker << " and " << *OtherAtom << "." << endl);
          }
        }
      }
    }
  }
  return count;
};

/** Counts the number of bonds between three given elements.
 * Note that we do not look for arbitrary sequence of given bonds, but \a *second will be the central atom and we check
 * whether it has bonds to both \a *first and \a *third.
 * \param *molecules list of molecules with all atoms
 * \param *first pointer to first element
 * \param *second pointer to second element
 * \param *third pointer to third element
 * \return number of found bonds (\a *first-\a *second-\a *third, \a *third-\a *second-\a *first, respectively)
 */
int CountBondsOfThree(MoleculeListClass * const molecules, const element * const first, const element * const second, const element * const third)
{
  int count = 0;
  bool MatchFlag[2];
  bool result = false;
  atom *Walker = NULL;
  const element * ElementArray[2];
  ElementArray[0] = first;
  ElementArray[1] = third;

  for (MoleculeList::const_iterator MolWalker = molecules->ListOfMolecules.begin();MolWalker != molecules->ListOfMolecules.end(); MolWalker++) {
    Walker = (*MolWalker)->start;
    while (Walker->next != (*MolWalker)->end) {
      Walker = Walker->next;
      if (Walker->type == second) {  // first element matches
        for (int i=0;i<2;i++)
          MatchFlag[i] = false;
        for (BondList::const_iterator BondRunner = Walker->ListOfBonds.begin(); BondRunner != Walker->ListOfBonds.end(); BondRunner++) {
          atom * const OtherAtom = (*BondRunner)->GetOtherAtom(Walker);
          for (int i=0;i<2;i++)
            if ((!MatchFlag[i]) && (OtherAtom->type == ElementArray[i])) {
              MatchFlag[i] = true;
              break;  // each bonding atom can match at most one element we are looking for
            }
        }
        result = true;
        for (int i=0;i<2;i++) // gather results
          result = result && MatchFlag[i];
        if (result) { // check results
          count++;
          DoLog(1) && (Log() << Verbose(1) << first->name << "-" << second->name << "-" << third->name << " bond found at " << *Walker << "." << endl);
        }
      }
    }
  }
  return count;
};
