/*
 * DataConverter.hpp
 *
 *  Created on: Aug 27, 2012
 *      Author: heber
 */

#ifndef DATACONVERTER_HPP_
#define DATACONVERTER_HPP_


// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <map>
#include <boost/fusion/sequence.hpp>

#include "CodePatterns/Assert.hpp"

#ifdef HAVE_JOBMARKET
#include "JobMarket/types.hpp"
#else
typedef size_t JobId_t;
#endif

#include "Fragmentation/KeySetsContainer.hpp"
#include "Fragmentation/Summation/SetValues/Eigenvalues.hpp"
#include "Fragmentation/Summation/SetValues/Histogram.hpp"
#include "Fragmentation/Summation/SetValues/Fragment.hpp"
#include "Fragmentation/Summation/SetValues/IndexedVectors.hpp"

#include "Fragmentation/Summation/Containers/MPQCData.hpp"
#include "Fragmentation/Summation/Containers/MPQCDataFused.hpp"
#include "Fragmentation/Summation/Containers/MPQCDataMap.hpp"
#ifdef HAVE_VMG
#include "Fragmentation/Summation/Containers/VMGData.hpp"
#include "Fragmentation/Summation/Containers/VMGDataFused.hpp"
#include "Fragmentation/Summation/Containers/VMGDataMap.hpp"
#endif

template <typename source, typename dest>
inline void convertDataTo(
    const std::map<JobId_t, source> &fragmentData,
    std::map<JobId_t, dest> &MPQCData_fused)
{
  MPQCData_fused.clear();
}

template <>
inline void convertDataTo<MPQCData, MPQCDataEnergyMap_t>(
    const std::map<JobId_t, MPQCData> &fragmentData,
    std::map<JobId_t, MPQCDataEnergyMap_t> &MPQCData_Energy_fused)
{
  // energy_t
  MPQCData_Energy_fused.clear();
  for(std::map<JobId_t, MPQCData>::const_iterator dataiter = fragmentData.begin();
      dataiter != fragmentData.end(); ++dataiter) {
    const MPQCData &extractedData = dataiter->second;
    LOG(4, "DEBUG: Current extracted Data is " << extractedData << ".");
    MPQCDataEnergyMap_t instance;
    boost::fusion::at_key<MPQCDataFused::energy_total>(instance) = extractedData.energies.total;
    boost::fusion::at_key<MPQCDataFused::energy_nuclear_repulsion>(instance) = extractedData.energies.nuclear_repulsion;
    boost::fusion::at_key<MPQCDataFused::energy_electron_coulomb>(instance) = extractedData.energies.electron_coulomb;
    boost::fusion::at_key<MPQCDataFused::energy_electron_exchange>(instance) = extractedData.energies.electron_exchange;
    boost::fusion::at_key<MPQCDataFused::energy_correlation>(instance) = extractedData.energies.correlation;
    boost::fusion::at_key<MPQCDataFused::energy_overlap>(instance) = extractedData.energies.overlap;
    boost::fusion::at_key<MPQCDataFused::energy_kinetic>(instance) = extractedData.energies.kinetic;
    boost::fusion::at_key<MPQCDataFused::energy_hcore>(instance) = extractedData.energies.hcore;
    boost::fusion::at_key<MPQCDataFused::energy_eigenvalues>(instance) = extractedData.energies.eigenvalues;
    boost::fusion::at_key<MPQCDataFused::energy_eigenhistogram>(instance) = extractedData.energies.eigenvalues;
    MPQCData_Energy_fused.insert( std::make_pair(dataiter->first, instance) );
  }
}

#ifdef HAVE_VMG
template <>
inline void convertDataTo<VMGData, VMGDataMap_t>(
    const std::map<JobId_t, VMGData> &longrangeData,
    std::map<JobId_t, VMGDataMap_t> &VMGData_fused)
{
  // energy_t
  VMGData_fused.clear();
  for(std::map<JobId_t, VMGData>::const_iterator dataiter = longrangeData.begin();
      dataiter != longrangeData.end(); ++dataiter) {
    const VMGData &extractedData = dataiter->second;
    LOG(4, "DEBUG: Current extracted Data is " << extractedData << ".");
    VMGDataMap_t instance;
    boost::fusion::at_key<VMGDataFused::nuclei_long>(instance) = extractedData.nuclei_long;
    boost::fusion::at_key<VMGDataFused::electron_long>(instance) = extractedData.electron_long;
    VMGData_fused.insert( std::make_pair(dataiter->first, instance) );
  }
}

template <>
inline void convertDataTo<VMGData, VMGDataGridMap_t>(
    const std::map<JobId_t, VMGData> &longrangeData,
    std::map<JobId_t, VMGDataGridMap_t> &VMGData_fused)
{
  // energy_t
  VMGData_fused.clear();
  for(std::map<JobId_t, VMGData>::const_iterator dataiter = longrangeData.begin();
      dataiter != longrangeData.end(); ++dataiter) {
    const VMGData &extractedData = dataiter->second;
    LOG(4, "DEBUG: Current extracted Data is " << extractedData << ".");
    VMGDataGridMap_t instance;
    boost::fusion::at_key<VMGDataFused::sampled_potential>(instance) = extractedData.sampled_potential;
    boost::fusion::at_key<VMGDataFused::both_sampled_potential>(instance) = extractedData.both_sampled_potential;
    VMGData_fused.insert( std::make_pair(dataiter->first, instance) );
  }
}
#endif

template <class datatype, class dataforcemap, class dataforcefused>
inline void convertDatatoForceMap(
    const std::map<JobId_t, datatype> &fragmentData,
    const KeySetsContainer &ForceKeySet,
    std::map<JobId_t, dataforcemap> &Data_Force_fused)
{
  // forces
  ASSERT( ForceKeySet.KeySets.size() == fragmentData.size(),
      "FragmentationAutomationAction::performCall() - indices and fragmentData differ in size.");
  Data_Force_fused.clear();
  typename std::map<JobId_t, datatype>::const_iterator dataiter = fragmentData.begin();
  KeySetsContainer::ArrayOfIntVectors::const_iterator arrayiter = ForceKeySet.KeySets.begin();
  for(;dataiter != fragmentData.end(); ++dataiter, ++arrayiter) {
    const datatype &extractedData = dataiter->second;
    LOG(4, "DEBUG: Current extracted Data is " << extractedData << ".");
    dataforcemap instance;
    // must convert int to index_t
    if (DoLog(5)) {
      std::stringstream output;
      for (KeySetsContainer::IntVector::const_iterator outiter = arrayiter->begin();
          outiter != arrayiter->end(); ++outiter) {
        output << *outiter << "\t";
      }
      LOG(5, "DEBUG: indices are " << output.str());
    }
    IndexedVectors::indices_t indices(arrayiter->begin(), arrayiter->end());
    boost::fusion::at_key<typename dataforcefused::forces>(instance) =
        IndexedVectors(indices, extractedData.forces);
    Data_Force_fused.insert( std::make_pair(dataiter->first, instance) );
  }
}

template <>
inline void convertDataTo<MPQCData, MPQCDataGridMap_t>(
    const std::map<JobId_t, MPQCData> &fragmentData,
    std::map<JobId_t, MPQCDataGridMap_t> &MPQCData_Grid_fused)
{
  // sampled_grid
  MPQCData_Grid_fused.clear();
  for(std::map<JobId_t, MPQCData>::const_iterator dataiter = fragmentData.begin();
      dataiter != fragmentData.end(); ++dataiter) {
    const MPQCData &extractedData = dataiter->second;
    LOG(4, "DEBUG: Current extracted Data is " << extractedData << ".");
    MPQCDataGridMap_t instance;
    boost::fusion::at_key<MPQCDataFused::sampled_grid>(instance) = extractedData.sampled_grid;
    MPQCData_Grid_fused.insert( std::make_pair(dataiter->first, instance) );
  }
}

template <>
inline void convertDataTo<MPQCData, MPQCDataFragmentMap_t>(
    const std::map<JobId_t, MPQCData> &fragmentData,
    std::map<JobId_t, MPQCDataFragmentMap_t> &MPQCData_Fragment_fused)
{
  // fragment
  MPQCData_Fragment_fused.clear();
  for(std::map<JobId_t, MPQCData>::const_iterator dataiter = fragmentData.begin();
      dataiter != fragmentData.end(); ++dataiter) {
    const MPQCData &extractedData = dataiter->second;
    LOG(4, "DEBUG: Current extracted Data is " << extractedData << ".");
    MPQCDataFragmentMap_t instance;
    boost::fusion::at_key<MPQCDataFused::fragment>(instance) =
        Fragment(extractedData.positions, extractedData.atomicnumbers, extractedData.charges);
    MPQCData_Fragment_fused.insert( std::make_pair(dataiter->first, instance) );
  }
}

template <>
inline void convertDataTo<MPQCData, MPQCDataTimeMap_t>(
    const std::map<JobId_t, MPQCData> &fragmentData,
    std::map<JobId_t, MPQCDataTimeMap_t> &MPQCData_Time_fused)
{
  // times
  MPQCData_Time_fused.clear();
  for(std::map<JobId_t, MPQCData>::const_iterator dataiter = fragmentData.begin();
      dataiter != fragmentData.end(); ++dataiter) {
    const MPQCData &extractedData = dataiter->second;
    LOG(4, "DEBUG: Current extracted Data is " << extractedData << ".");
    MPQCDataTimeMap_t instance;
    boost::fusion::at_key<MPQCDataFused::times_total_walltime>(instance) = extractedData.times.total_walltime;
    boost::fusion::at_key<MPQCDataFused::times_total_cputime>(instance) = extractedData.times.total_cputime;
    boost::fusion::at_key<MPQCDataFused::times_total_flops>(instance) = extractedData.times.total_flops;
    boost::fusion::at_key<MPQCDataFused::times_gather_walltime>(instance) = extractedData.times.gather_walltime;
    boost::fusion::at_key<MPQCDataFused::times_gather_cputime>(instance) = extractedData.times.gather_cputime;
    boost::fusion::at_key<MPQCDataFused::times_gather_flops>(instance) = extractedData.times.gather_flops;
    MPQCData_Time_fused.insert( std::make_pair(dataiter->first, instance) );
  }
}


#endif /* DATACONVERTER_HPP_ */
