/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2016 Frederik Heber. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * FragmentForcesUnitTest.cpp
 *
 *  Created on: Aug 18, 2016
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

using namespace std;

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

// include headers that implement a archive in simple text format
#include <boost/archive/text_oarchive.hpp>
#include <boost/archive/text_iarchive.hpp>

#include "FragmentForcesUnitTest.hpp"

#include <algorithm>
#include <limits>

#include <boost/assign.hpp>
#include <boost/foreach.hpp>

#include <sstream>

#include "CodePatterns/Assert.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

using namespace boost::assign;

#include "Fragmentation/Summation/SetValues/FragmentForces.hpp"

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( FragmentForcesTest );


void FragmentForcesTest::setUp()
{
  // failing asserts should be thrown
  ASSERT_DO(Assert::Throw);
}

void FragmentForcesTest::tearDown()
{
}

/** UnitTest for operator==()
 */
void FragmentForcesTest::equality_Test()
{
  // create different forces
  FragmentForces::force_t zeroforce;
  zeroforce += 0., 0., 0.;
  FragmentForces::force_t oneforce;
  oneforce += 1., 2., 3.;
  FragmentForces::force_t twoforce;
  twoforce += 0., 1., 4.;

  FragmentForces emptyfragment;
  FragmentForces fragment;
  fragment += oneforce;
  fragment += twoforce;
  FragmentForces samefragment;
  samefragment += oneforce;
  samefragment += twoforce;
  FragmentForces otherfragment;
  otherfragment += twoforce;
  otherfragment += twoforce;
  FragmentForces largerfragment;
  largerfragment += oneforce;
  largerfragment += oneforce;
  largerfragment += oneforce;
  FragmentForces smallerfragment;
  smallerfragment += oneforce;

  // test against different instance with same contents
  CPPUNIT_ASSERT( (fragment == samefragment) );
  CPPUNIT_ASSERT( !(fragment != samefragment) );

  // test against other fragment
  CPPUNIT_ASSERT( !(fragment == otherfragment) );
  CPPUNIT_ASSERT( fragment != otherfragment );

  // test against larger fragment
  CPPUNIT_ASSERT( !(fragment == largerfragment) );
  CPPUNIT_ASSERT( fragment != largerfragment );

  // test against smaller fragment
  CPPUNIT_ASSERT( !(fragment == smallerfragment) );
  CPPUNIT_ASSERT( fragment != smallerfragment );

  // test against empty
  CPPUNIT_ASSERT( !(fragment == emptyfragment) );
  CPPUNIT_ASSERT( fragment != emptyfragment );

  // tests against themselves
  CPPUNIT_ASSERT( fragment == fragment );
  CPPUNIT_ASSERT( samefragment == samefragment );
  CPPUNIT_ASSERT( otherfragment == otherfragment );
  CPPUNIT_ASSERT( largerfragment == largerfragment );
  CPPUNIT_ASSERT( smallerfragment == smallerfragment );
  CPPUNIT_ASSERT( emptyfragment == emptyfragment );
}

/** UnitTest for operator+=()
 */
void FragmentForcesTest::operatorPlusEqual_Test()
{
  // create different forces
  FragmentForces::force_t zeroforce;
  zeroforce += 0., 0., 0.;
  FragmentForces::force_t oneforce;
  oneforce += 1., 2., 3.;
  FragmentForces::force_t twoforce;
  twoforce += 0., 1., 4.;
  FragmentForces::force_t resultforce;
  resultforce += 1., 3., 7.;

  FragmentForces fragment;
  fragment += oneforce;
  fragment += oneforce;
  FragmentForces otherfragment;
  otherfragment += twoforce;
  otherfragment += twoforce;
  FragmentForces resultfragment;
  resultfragment += resultforce;
  resultfragment += resultforce;

  FragmentForces sumfragment = fragment;
  sumfragment.operator+=(otherfragment);
  CPPUNIT_ASSERT( sumfragment == resultfragment);
}

/** UnitTest for operator+=() with different number of components
 */
void FragmentForcesTest::operatorPlusEqual_NonOverlapping_Test()
{
  // create different forces
  FragmentForces::force_t zeroforce;
  zeroforce += 0., 0., 0.;
  FragmentForces::force_t surplusforce;
  surplusforce += 0., 0., 0., 1.;
  FragmentForces::force_t oneforce;
  oneforce += 1., 2., 3.;
  FragmentForces::force_t twoforce;
  twoforce += 0., 1., 4.;
  FragmentForces::force_t resultforce;
  resultforce += 1., 3., 7.;

  FragmentForces fragment;
  fragment += oneforce;
  fragment += oneforce;
  FragmentForces otherfragment;
  otherfragment += twoforce;
  FragmentForces surplusfragment;
  surplusfragment += surplusforce;
  surplusfragment += twoforce;

  {
    FragmentForces sumfragment = fragment;
#ifndef NDEBUG
    std::cout << "The following assertion is expected and does not mean a failure of the test." << std::endl;
    CPPUNIT_ASSERT_THROW( (sumfragment.operator+=(otherfragment)) ,Assert::AssertionFailure);
#endif
  }
  {
    FragmentForces sumfragment = fragment;
#ifndef NDEBUG
    std::cout << "The following assertion is expected and does not mean a failure of the test." << std::endl;
    CPPUNIT_ASSERT_THROW( (sumfragment.operator+=(surplusfragment)) ,Assert::AssertionFailure);
#endif
  }
}
