/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2014 Frederik Heber. All rights reserved.
 *
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * FragmentForces.cpp
 *
 *  Created on: Oct 23, 2014
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// include headers that implement a archive in simple text format
// otherwise BOOST_CLASS_EXPORT_IMPLEMENT has no effect
#include <boost/archive/text_oarchive.hpp>
#include <boost/archive/text_iarchive.hpp>

#include "CodePatterns/MemDebug.hpp"

#include "CodePatterns/Assert.hpp"

#include "FragmentForces.hpp"

bool FragmentForces::operator==(const FragmentForces& other) const
{
  if( size() != other.size())
    return false;

  forces_t::const_iterator iter = begin();
  forces_t::const_iterator otheriter = other.begin();
  for (; iter != end(); ++iter, ++otheriter) {
    if( iter->size() != otheriter->size())
      return false;
    force_t::const_iterator forceiter = iter->begin();
    force_t::const_iterator otherforceiter = otheriter->begin();
    for (; forceiter != iter->end(); ++forceiter, ++otherforceiter) {
      if (*forceiter != *otherforceiter)
        return false;
    }
  }
  return true;
}

void FragmentForces::operator+=(const FragmentForces& other)
{
  ASSERT( size() == other.size(),
      "FragmentForces::operator+=() - unequal number of force vectors.");
  forces_t::iterator iter = begin();
  forces_t::const_iterator otheriter = other.begin();
  for (; iter != end(); ++iter, ++otheriter) {
    ASSERT( iter->size() == otheriter->size(),
        "FragmentForces::operator+=() - force vectors having unequal number of components.");
    force_t::iterator forceiter = iter->begin();
    force_t::const_iterator otherforceiter = otheriter->begin();
    for (; forceiter != iter->end(); ++forceiter, ++otherforceiter) {
      *forceiter += *otherforceiter;
    }
  }
}

std::ostream & operator<<(std::ostream &ost, const FragmentForces &forces)
{

  for (FragmentForces::forces_t::const_iterator iter = forces.begin();
      iter != forces.end(); ++iter) {

    if (iter != forces.begin())
      ost << "; ";
    ost << "[";
    for (FragmentForces::force_t::const_iterator forceiter = iter->begin();
        forceiter != iter->end(); ++forceiter) {
      if (forceiter != iter->begin())
        ost << ",";
      ost << *forceiter;
    }
    ost << "]";
  }
  return ost;
}

// we need to explicitly instantiate the serialization functions as
// its is only serialized through its base class FragmentJob
BOOST_CLASS_EXPORT_IMPLEMENT(FragmentForces)
