/*
 * AllLevelSummator.hpp
 *
 *  Created on: 29.07.2012
 *      Author: heber
 */

#ifndef ALLLEVELSUMMATOR_HPP_
#define ALLLEVELSUMMATOR_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <vector>

#include "CodePatterns/Assert.hpp"

#include "Fragmentation/Summation/IndexSetContainer.hpp"
#include "Fragmentation/Summation/SubsetMap.hpp"
#include "Fragmentation/Summation/Summator.hpp"

#include "Fragmentation/Summation/printKeyNames.hpp"

/** Tiny template functor to use Summation, sum up each level, and store the
 * result in a given vector.
 *
 */
template <typename MapType>
struct AllLevelSummator {
  /** Constructor takes the arguments that \a Summator also needs and stores
   * them internally.
   *
   * \param _subsetmap map with hierarchy of IndexSet's
   * \param _data MPQCData converted to MPQCDataMap_t type, associated to job id
   * \param _container container of IndexSet's such that each set has correct order
   *        to job id and hence to _data.
   * \param _MatrixNrLookup lookup from job id to ordering in above vectors, if
   *        out of range this value is set to zero, hence ignored.
   * \param _levelresults vector place levelresults into
   */
  AllLevelSummator(
      SubsetMap::ptr &_subsetmap,
      const std::map<JobId_t, MapType> &_data,
      const IndexSetContainer::Container_t &_container,
      const std::map< JobId_t, size_t > &_MatrixNrLookup,
      std::vector<MapType> &_levelresults,
      std::map<IndexSet::ptr, std::pair<MapType, MapType> > &_keysetresults) :
    subsetmap(_subsetmap),
    data(_data),
    container(_container),
    MatrixNrLookup(_MatrixNrLookup),
    levelresults(_levelresults),
    keysetresults(_keysetresults)
  {
    ASSERT( levelresults.size() >= subsetmap->getMaximumSetLevel(),
        "AllLevelSummator() - result vector is not large enough.");
  }

  /** Operator that calls on Summator and prints the value.
   *
   * \note the parameter is needed for boost::mpl::for_each but is not
   * used here.
   */
  template <typename MapKey>
  void operator()(MapKey &) {
    // We retrieve the type of the MPQCData member variable from the boost::fusion::map.
    typedef typename boost::fusion::result_of::value_at_key<MapType, MapKey>::type MapValue;

    // create Summator instance
    Summator<MapType, MapKey> sum_value(
        subsetmap, data, container, MatrixNrLookup
        );

    // fill levelresults
    const size_t MaxLevel = subsetmap->getMaximumSetLevel();
    for (size_t level=1; level <= MaxLevel; ++level) {
      MapType &LevelResults = levelresults[level-1];
      // sum up and store in levelresults
      const MapValue value = sum_value(level);
      boost::fusion::at_key<MapKey>(LevelResults) = value;
      // print value
      //LOG(0, "STATUS: Level " << level << " resulting " << printKeyNames::printName<MapKey>() << " is " << value << ".");
    }

    // fill keysetresults
    for (IndexSetContainer::Container_t::const_iterator indexsetiter = container.begin();
        indexsetiter != container.end(); ++indexsetiter) {
      const IndexSet::ptr &index = *indexsetiter;
      // this either generates a new entry or updates the present one, as we obtain ref to value
      boost::fusion::at_key<MapKey>(keysetresults[index].first) =
          sum_value.getValueForIndexSet(index);
      boost::fusion::at_key<MapKey>(keysetresults[index].second) =
          sum_value.getContributionForIndexSet(index);
    }
  }

private:
  //!> Hierarchy of IndexSet's
  SubsetMap::ptr &subsetmap;
  //!> vector of data converted from MPQCData
  const std::map<JobId_t, MapType> &data;
  //!> container with all IndexSet's
  const IndexSetContainer::Container_t &container;
  //!> lookup map from job ids to ordering in above vectors
  const std::map< JobId_t, size_t > &MatrixNrLookup;
  //!> vector of levelresults
  std::vector<MapType> &levelresults;
  typedef std::pair< IndexSet::ptr, std::pair<MapType, MapType> > keysetresults_pair_t;
  //!> typedef for map for keyset and each resulting value
  typedef std::map<IndexSet::ptr, std::pair<MapType, MapType> > keysetresults_t;
  //!> map for IndexSet and its associated contribution
  keysetresults_t &keysetresults;
};

#endif /* ALLLEVELSUMMATOR_HPP_ */
