/*
 * ObservedValuesContainer.hpp
 *
 *  Created on: Oct 29, 2015
 *      Author: heber
 */


#ifndef OBSERVEDVALUESCONTAINER_HPP_
#define OBSERVEDVALUESCONTAINER_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "ObservedValue_types.hpp"

#include <map>
#include <string>

class QtObservedInstanceBoard;

/** This class contains ObservedValues of the class \b T each instance identified
 * by the id type \b id.
 *
 * All the reference counting is done inside this container.
 */
template <class T, typename id>
class ObservedValuesContainer
{
  /** Cstor of class ObservedValuesContainer.
   *
   * \param _name name used in debugging and prints
   * \param _board ref to InstanceBoard
   */
  ObservedValuesContainer(
      const std::string _name,
      QtObservedInstanceBoard &_board);

  /** Delivers the set of Observed value for the instance identified by \a _id.
   *
   * \param _id identifier of the instance
   * \return shared ptr to observed instance.
   */
  typename T::ptr get(const id _id);

  /** Returns an observed instance identified by \a _id, for deallocation.
   *
   * \param _id identifier of the instance
   */
  void yield(const id _id);

  ObservedValues_t getObservedValues(const id _id);

  void returnObservedValues(const id _id, ObservedValues_t &_observedvalues);

private:
  typedef std::pair<ObservedValues_t, size_t> RefCountedObservedValues_t;
  typedef std::map<id, RefCountedObservedValues_t> CountedObservedValues_t;
  //!> internal vector of observed values
  CountedObservedValues_t ObservedValues;

  //!> name used in describing the instance type
  const std::string NameOfType;

  //!> reference to InstanceBoard for callbacks on subjectKilled()
  QtObservedInstanceBoard &board;

private:
  //!> QtObservedInstanceBoard may access anything
  friend class QtObservedInstanceBoard;

  /** Inserts a new ObservedValue vector into the container.
   *
   * \param _id identifier of instance associated with observed values
   * \param _obsvalues vector of observed values of instance
   * \return true - insertion successful, false - else
   */
  bool insert(const id _id, ObservedValues_t &_obsvalues);

  /** Use to change the identifier associated with a vector of observed values.
   *
   * \param _oldid old identifier
   * \param _newid new identifier
   * \return true - change successful, false - else
   */
  bool changeIdentifier(const id _oldid, const id _newid);

  /** Returns the number of reference delivered for this instance identified by \a _id.
   *
   * \param _id identifier of instance
   * \return reference count
   */
  size_t getRefCount(const id _id) const;

  /** Erases a vector of observed values of an instance  identified by \a _id.
   *
   * \param _id identifier of instance
   * \return true - erase successful, false - else
   */
  bool erase(const id _id);

  /** Checks whether a vector of observed values of an instance  identified by \a _id
   * is present.
   *
   * \param _id identifier of instance
   * \return true - present, false - else
   */
  bool isPresent(const id _id) const;
};

#endif /* OBSERVEDVALUESCONTAINER_HPP_ */
