/*
 * SphericalPointDistribution.hpp
 *
 *  Created on: May 29, 2014
 *      Author: heber
 */


#ifndef SPHERICALPOINTDISTRIBUTION_HPP_
#define SPHERICALPOINTDISTRIBUTION_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/Assert.hpp"

#include <cmath>
#include <list>

#include "LinearAlgebra/Vector.hpp"

/** contains getters for the VSEPR model for specific number of electrons.
 *
 * This struct contains specialized functions returning a list of Vectors
 * (points in space) to match the VSEPR model for the given number of electrons.
 *
 * This is implemented via template specialization of the function get().
 *
 * These specializations are taken from the python script \b CreateVspeShapes.py
 * by Christian Neuen, 07th May 2009.
 */
struct SphericalPointDistribution
{
  /** Cstor for SphericalPointDistribution, allows setting radius of sphere
   *
   * \param _BondLength desired radius of sphere
   */
  SphericalPointDistribution(const double _Bondlength = 1.) :
    Bondlength(_Bondlength),
    SQRT_3(sqrt(3.0))
  {}

  //!> typedef for the list of points
  typedef std::list<Vector> Polygon_t;

  /** General getter function for the distribution of points on the surface.
   *
   * \warn this function needs to be specialized!
   *
   * \return Polygon_t with points on the surface centered at (0,0,0)
   */
  template <int N> Polygon_t get()
  {
    ASSERT(0, "SphericalPointDistribution::get() - not specialized for "+toString(N)+".");
  }


  /** Matches a given spherical distribution with another containing more
   * points.
   *
   * This is a helper to determine points where to best insert saturation
   * hydrogens.
   *
   * \param _polygon current occupied positions
   * \param _newpolygon ideal distribution to match best with current occupied
   *        positions
   * \return remaining vacant positions relative to \a _polygon
   */
  static Polygon_t matchSphericalPointDistributions(
      const Polygon_t &_polygon,
      const Polygon_t &_newpolygon
      );


  //!> default radius of the spherical distribution
  const double Bondlength;
  //!> precalculated value for root of 3
  const double SQRT_3;
};

// declare specializations

template <> SphericalPointDistribution::Polygon_t SphericalPointDistribution::get<0>();
template <> SphericalPointDistribution::Polygon_t SphericalPointDistribution::get<1>();
template <> SphericalPointDistribution::Polygon_t SphericalPointDistribution::get<2>();
template <> SphericalPointDistribution::Polygon_t SphericalPointDistribution::get<3>();
template <> SphericalPointDistribution::Polygon_t SphericalPointDistribution::get<4>();
template <> SphericalPointDistribution::Polygon_t SphericalPointDistribution::get<5>();
template <> SphericalPointDistribution::Polygon_t SphericalPointDistribution::get<6>();
template <> SphericalPointDistribution::Polygon_t SphericalPointDistribution::get<7>();
template <> SphericalPointDistribution::Polygon_t SphericalPointDistribution::get<8>();
template <> SphericalPointDistribution::Polygon_t SphericalPointDistribution::get<9>();
template <> SphericalPointDistribution::Polygon_t SphericalPointDistribution::get<10>();
template <> SphericalPointDistribution::Polygon_t SphericalPointDistribution::get<11>();
template <> SphericalPointDistribution::Polygon_t SphericalPointDistribution::get<12>();
template <> SphericalPointDistribution::Polygon_t SphericalPointDistribution::get<14>();

#endif /* SPHERICALPOINTDISTRIBUTION_HPP_ */
