/*
 * FunctionArgument.hpp
 *
 *  Created on: 02.10.2012
 *      Author: heber
 */

#ifndef FUNCTIONARGUMENT_HPP_
#define FUNCTIONARGUMENT_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <utility>
#include <iosfwd>

/** This class encapsulates all information with respect to a single argument
 *  for a high-dimensional model function.
 *
 *  We restrict ourselves here to a function that dependent on a set of
 *  three-dimensional vectors, i.e. a set of positions in space. And for
 *  the moment to distances in between these sets.
 *
 */
struct argument_t
{
  //!> grant operator access to private parts
  friend std::ostream& operator<<(std::ostream &ost, const argument_t &arg);

  //!> typedef for the two indices of the argument
  typedef std::pair<size_t, size_t> indices_t;
  //!> typedef for the underlying type of the particle
  typedef int ParticleType_t;
  //!> typedef for the two particle types of the argument
  typedef std::pair<ParticleType_t, ParticleType_t> types_t;

  /** Default constructor for class argument_t.
   *
   */
  argument_t() :
    indices( std::make_pair(0,1) ),
    types( std::make_pair(0,0) ),
    distance(0.),
    globalid(-1)
  {}

  /** Constructor for class argument_t.
   *
   * This constructors uses the index pair (0,1) as default.
   *
   * \param _distance distance argument
   */
  argument_t(const double &_distance) :
    indices( std::make_pair(0,1) ),
    types( std::make_pair(0,0) ),
    distance(_distance),
    globalid(-1)
  {}

  /** Constructor for class argument_t.
   *
   * \param _indices pair of indices associated with the \a _distance
   * \param _distance distance argument
   */
  argument_t(const indices_t &_indices, const double &_distance) :
    indices( _indices ),
    types( std::make_pair(0,0) ),
    distance(_distance),
    globalid(-1)
  {}

  /** Constructor for class argument_t.
   *
   * \param _indices pair of indices associated with the \a _distance
   * \param _types pair of particle type
   * \param _distance distance argument
   */
  argument_t(const indices_t &_indices, const types_t &_types, const double &_distance) :
    indices( _indices ),
    types( _types ),
    distance(_distance),
    globalid(-1)
  {}

  /** Comparator with respect to the distance.
   *
   * \note We'll have this as static function to allow usage in e.g. STL's sort.
   *
   * \param one first argument
   * \param other other argument to compare to \a one to
   * \return true - first distance is less
   */
  static bool DistanceComparator(const argument_t &one, const argument_t &other)
  {
    return one.distance < other.distance;
  }

  /** Comparator with respect to the pair of types.
   *
   * \note We'll have this as static function to allow usage in e.g. STL's sort.
   *
   * \param one first argument
   * \param other other argument to compare to \a one to
   * \return true - first type is less or if equal, second type is less, else
   */
  bool static TypeComparator(const argument_t &one, const argument_t &other)
  {
    if (one.types.first < other.types.first)
      return true;
    else if (one.types.first > other.types.first)
      return false;
    else
      return one.types.second < other.types.second;
  }

  /** Comparator with respect to the pair of indices.
   *
   * \note We'll have this as static function to allow usage in e.g. STL's sort.
   *
   * \param one first argument
   * \param other other argument to compare to \a one to
   * \return true - first index is less or if equal, second index is less, else
   */
  bool static IndexComparator(const argument_t &one, const argument_t &other)
  {
    if (one.indices.first < other.indices.first)
      return true;
    else if (one.indices.first > other.indices.first)
      return false;
    else
      return one.indices.second < other.indices.second;
  }

  /** Less comparator for FunctionArgument.
   *
   * @param other other argument to compare to
   * @return true - this argument is less than \a other, false - else
   */
  bool operator<(const argument_t &other) const
  {
    if (types.first < other.types.first)
      return true;
    else if (types.first > other.types.first)
      return false;
    else if (types.second < other.types.second)
      return true;
    else if (types.second > other.types.second)
      return false;
    else {
      if (indices.first < other.indices.first)
        return true;
      else if (indices.first > other.indices.first)
        return false;
      else
        return indices.second < other.indices.second;
    }
  }

  /** Equality operator for FunctionArgument.
   *
   * \note This compares only types and indices.
   *
   * @param other other argument to compare to
   * @return true - this argument is equal to \a other, false - else
   */
  bool operator==(const argument_t &other) const
  {
    if (types.first != other.types.first)
      return false;
    else if (types.second != other.types.second)
      return false;
    if (indices.first != other.indices.first)
      return false;
    else if (indices.second != other.indices.second)
      return false;
    else
      return true;
  }

  //!> indices between which the distance is given
  indices_t indices;
  //!> indices between which the distance is given
  types_t types;
  //!> distance
  double distance;
  //!> global id refers to some global index, e.g. the configuration id in training set
  size_t globalid;
};

/** Print given \a arg to stream \a ost.
 *
 * \param ost output stream to print to
 * \param arg argument to print
 * \return output stream for concatenation
 */
std::ostream& operator<<(std::ostream &ost, const argument_t &arg);


#endif /* FUNCTIONARGUMENT_HPP_ */
