/*
 * GLWorldScene.hpp
 *
 *  This is based on the Qt3D example "teaservice", specifically parts of teaservice.cpp.
 *
 *  Created on: Aug 17, 2011
 *      Author: heber
 */

#ifndef GLWORLDSCENE_HPP_
#define GLWORLDSCENE_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <Qt/qobject.h>

#include <iosfwd>

#include <boost/bimap.hpp>
#include <boost/thread/recursive_mutex.hpp>

#include "Bond/bond.hpp"
#include "GLMoleculeObject_bond.hpp"
#include "GLMoleculeObject_molecule.hpp"
#include "types.hpp"

#include "UIElements/Qt4/InstanceBoard/QtObservedAtom.hpp"
#include "UIElements/Qt4/InstanceBoard/QtObservedBond.hpp"
#include "UIElements/Qt4/InstanceBoard/QtObservedMolecule.hpp"

class Shape;

class QGLPainter;
class QGLSceneNode;
class QGLView;

class GLMoleculeObject;
class GLMoleculeObject_atom;
class GLMoleculeObject_molecule;
class GLMoleculeObject_shape;

class QtObservedInstanceBoard;

/** This class contains a list of all molecules in the world.
 *
 */
class GLWorldScene : public QObject
{
  Q_OBJECT
public:
  GLWorldScene(
      QtObservedInstanceBoard * _board,
      QObject *parent=0);
  virtual ~GLWorldScene();

//#if !defined(QT_OPENGL_ES_1)
//  PerPixelEffect *lighting;
//#endif

  void changeMaterials(bool perPixel);
  QGLSceneNode* getAtom(size_t);
  QGLSceneNode* getMolecule(size_t);
  QGLSceneNode* getBond(size_t, size_t);

  void initialize(QGLView *view, QGLPainter *painter) const;
  void draw(QGLPainter *painter, const QVector4D &cameraPlane) const;

  enum SelectionModeType{
    SelectAtom,
    SelectMolecule
  };
  void setSelectionMode(SelectionModeType mode);

signals:
  void changed();
//  void updated();
  void changeOccured();
  void pressed();
  void released();
  void clicked();
  void clicked(atomId_t);
  void doubleClicked();
  void hoverChanged(const atomId_t);
  void hoverChanged(const moleculeId_t, int);
  void atomConnected(QtObservedAtom::ptr);
  void bondConnected(QtObservedBond::ptr);

private slots:

  void clickAtom(atomId_t no);
  void connectAtom(QtObservedAtom::ptr);
  void insertAtom(QtObservedAtom::ptr);
  void removeAtom(ObservedValue_Index_t _atomid);

  void connectBond(QtObservedBond::ptr);
  void insertBond(QtObservedBond::ptr);
  void removeBond(ObservedValue_Index_t _bondid);

  void clickMolecule(moleculeId_t no);
  void insertMolecule(QtObservedMolecule::ptr);
  void removeMolecule(ObservedValue_Index_t _molid);

  void setSelectionModeAtom();
  void setSelectionModeMolecule();

  void addShape(const std::string &_name);
  void removeShape(const std::string &_name);
//  void update();
  void moleculesVisibilityChanged(ObservedValue_Index_t _id, bool _visible);
  void hoverChangedSignalled(GLMoleculeObject *ob);

  void reparentAtom();
  void reparentBondLeft();
  void reparentBondRight();

private:

  GLMoleculeObject_molecule *getMoleculeObject(
      const ObservedValue_Index_t _molid) const;

  void resetParent(
      GLMoleculeObject *_obj,
      GLMoleculeObject_molecule *_molobj);

  void reparentBond(
      const QtObservedBond::ptr _bond,
      const QtObservedAtom::ptr _atom,
      const GLMoleculeObject_bond::SideOfBond _side);

  GLMoleculeObject_bond *getBondInScene(
      const ObservedValue_Index_t _bondid,
      GLMoleculeObject_bond::SideOfBond _side) const;

public:
  void updateSelectedShapes();

  typedef std::map< QObject*, QtObservedAtom::ptr > ObservedAtoms_t;
  typedef std::map< QObject*, QtObservedBond::ptr > ObservedBonds_t;

  typedef std::map< ObservedValue_Index_t, QObject* > ObservedValueIndexLookup_t;

private:

  //!> id of atom hovered over
  atomId_t hoverAtomId;

  typedef std::map< ObservedValue_Index_t, GLMoleculeObject_molecule* > MoleculeNodeMap;
  typedef std::map< ObservedValue_Index_t, GLMoleculeObject_atom* > AtomNodeMap;
  typedef std::multimap< ObservedValue_Index_t, GLMoleculeObject_bond* > BondNodeMap;
  typedef std::map< std::string , GLMoleculeObject_shape* > ShapeNodeMap;

  //!> typedef for storing the current parent to each atom node in scene, NULL indicates GLWorldScene is parent
  typedef boost::bimap<
      boost::bimaps::set_of< ObservedValue_Index_t >,
      boost::bimaps::multiset_of< ObservedValue_Index_t >
  > AtomNodeParentMap_t;
  //!> typedef for storing the current parent to each bond node in scene, NULL indicates GLWorldScene is parent
  typedef boost::bimap<
      boost::bimaps::set_of< ObservedValue_Index_t >,
      boost::bimaps::multiset_of< ObservedValue_Index_t >
  > BondNodeParentMap_t;

  //!> mutex to ascertain atomic access to NodeParentMaps
  boost::recursive_mutex AtomNodeParentMap_mutex;
  boost::recursive_mutex BondNodeParentMap_mutex;

  AtomNodeMap AtomsinSceneMap;
  BondNodeMap BondsinSceneMap;
  MoleculeNodeMap MoleculesinSceneMap;
  ShapeNodeMap ShapesinSceneMap;
  AtomNodeParentMap_t AtomNodeParentMap;
  std::vector<BondNodeParentMap_t> BondNodeParentMaps;

  //!> stored observed atom such that its signals may always be safely evaluated
  ObservedAtoms_t ObservedAtoms;
  //!> stored observed bond such that its signals may always be safely evaluated
  ObservedBonds_t ObservedBonds;
  //!> in moleculeChanged() we only have QObject*, in reparent..() we have ObservedValue_Index_t, need lookup
  ObservedValueIndexLookup_t ObservedValueIndexLookup;

  SelectionModeType selectionMode;

  QtObservedInstanceBoard * board;
};

#endif /* GLWORLDSCENE_HPP_ */
