/*
 * Value.hpp
 *
 *  Created on: Apr 13, 2012
 *      Author: ankele
 */

#ifndef VALUE_HPP_
#define VALUE_HPP_


// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string>
#include <vector>

#include "ValueAsString.hpp"
#include "ValueInterface.hpp"
#include "Validators/Validator.hpp"

#include "StreamOperators.hpp"

#include "CodePatterns/Range.hpp"
#include "CodePatterns/toString.hpp"

class ValueTest;
template <class T>
class Parameter;
class ParameterException;
class ParameterValidatorException;
class ParameterValueException;

/** Converter for a string to a std::vector of any class
 * We use default conversion via stringstream as suggested by [Stroustrup].
 * \param _&_object reference to object to convert.
 * \return converted \a _object of templated type
 */
template <class T>
struct ConvertTo< std::vector<T> > {
  std::vector<T> operator()(const std::string &_object) {
    std::vector<T> returnobjects;
    std::stringstream s;
    T object;
    s << _object;
    while (s.good()) {
      s >> object;
      returnobjects.push_back(object);
    }
    return returnobjects;
  }
};

/** This class represents a general value.
 *
 */
template <class T>
class Value :
  virtual public ValueAsString,
  public ValueInterface<T>
{
  //!> unit test needs to have access to internal values
  friend class ValueTest;
  friend class ContinuousValueTest;
  friend class Parameter<T>;
public:
  Value();
  Value(const Validator<T> &_validator);
  Value(const std::vector<T> &_ValidValues);
  Value(const range<T> &_ValidRange);
  virtual ~Value();

  // functions for ValueInterface
  bool isValid(const T &_value) const throw(ParameterValidatorException);
  const T & get() const throw(ParameterValueException);
  void set(const T & _value) throw(ParameterException);
  bool isSet() const;

  // string functions for ValueInterface
  bool isValidAsString(const std::string &_value) const throw(ParameterValidatorException);
  const std::string getAsString() const throw(ParameterValueException);
  void setAsString(const std::string &_value) throw(ParameterException);

  // comfortable setter
  Value<T> &operator=(const T &_value)
  { set(_value);  return *this;  }

  // comparator
  bool operator==(const Value<T> &_instance) const throw(ParameterValidatorException);
  bool operator!=(const Value<T> &_instance) const throw(ParameterValidatorException)
      { return !((*this)==(_instance)); }

  const Validator<T> & getValidator() const;
  Validator<T> & getValidator();

  // comfortable validator functions
  const range<T> & getValidRange() const throw(ParameterValidatorException);
  void setValidRange(const range<T> &_range) throw(ParameterValueException);
  void appendValidValue(const T &_value) throw(ParameterValidatorException);
  const std::vector<T> &getValidValues() const throw(ParameterValidatorException);

private:
  //!> Internal converter from string to internal type
  static ConvertTo<T> Converter;

  //!> whether a value has been set or not
  bool ValueSet;
  //!> contained value
  T value;

  //!> the validator
  Validator<T> *validator;
};

#include "Value_impl.hpp"

#include "Value_string.hpp"

#endif /* VALUE_HPP_ */
